package utility.valve;

/**
 * 阀门选型环境类
 */
public class ValvePot {
    private ValveUnit.PUnit P_unit;//默认压力单位
    private ValveUnit.QUnit Q_unit;//默认流量单位
    private ValveUnit.VUnit V_unit;//默认流速单位
    private double Pa = 10.339;//当地大气压，默认单位mH2O
    private double Pv = 0.23;//饱和蒸汽压，默认单位mH2O
    private double g = 9.81;//重力加速度
    private double ρ = 1.0;//密度,默认为水的密密度，1吨每立方
    private Medium medium;

    enum Medium {
        liquid, gas
    }

    public ValvePot(Medium medium) {
        this.medium = medium;
        this.P_unit = ValveUnit.PUnit.mH2O;
        this.Q_unit = ValveUnit.QUnit.m3h;
        this.V_unit = ValveUnit.VUnit.ms;
    }

    public ValvePot() {
        this.medium = Medium.liquid;
        this.P_unit = ValveUnit.PUnit.mH2O;
        this.Q_unit = ValveUnit.QUnit.m3h;
        this.V_unit = ValveUnit.VUnit.ms;
    }

    /**
     * @param p_unit 压力单位
     */
    public ValvePot(ValveUnit.PUnit p_unit, ValveUnit.QUnit q_unit, ValveUnit.VUnit v_unit) {
        this.medium = Medium.liquid;
        this.P_unit = p_unit;
        this.Q_unit = q_unit;
        this.V_unit = v_unit;
    }

    /**
     * @param p_unit 压力单位
     */
    public ValvePot(Medium medium, ValveUnit.PUnit p_unit, ValveUnit.QUnit q_unit, ValveUnit.VUnit v_unit) {
        this.medium = medium;
        this.P_unit = p_unit;
        this.Q_unit = q_unit;
        this.V_unit = v_unit;
    }

    /**
     * 获取默认压力单位
     *
     * @return
     */
    public ValveUnit.PUnit getP_unit() {
        return this.P_unit;
    }

    /**
     * 获取默认流量单位
     *
     * @return
     */
    public ValveUnit.QUnit getQ_unit() {
        return this.Q_unit;
    }

    public ValveUnit.VUnit getV_unit() {
        return this.V_unit;
    }

    /**
     * 设置当地大气压，
     * 、
     *
     * @param pa
     */
    public void setPa(double pa) {
        this.Pa = pa;
    }

    /**
     * 获取当地大气压
     *
     * @param pUnit
     * @return
     */
    public double getPa(ValveUnit.PUnit pUnit) {
        return ValveUnit.getPUnitValue(P_unit, Pa, pUnit);
    }


    /**
     * 设置饱和蒸气压，
     *
     * @param pv
     */
    public void setPv(double pv) {
        this.Pv = pv;
    }

    /**
     * 获取饱和蒸气压
     *
     * @param pUnit
     * @return
     */
    public double getPv(ValveUnit.PUnit pUnit) {
        return ValveUnit.getPUnitValue(P_unit, Pv, pUnit);
    }


    /**
     * 设置重力加速度
     *
     * @param g
     */
    public void setG(double g) {
        this.g = g;
    }

    /**
     * 获取重力加速度
     *
     * @return
     */
    public double getG() {
        return g;
    }


    /**
     * 设置介质密度
     *
     * @param ρ
     */
    public void setρ(double ρ) {
        this.ρ = ρ;
    }

    /**
     * 获取介质密度
     *
     * @return
     */
    public double getρ() {
        return ρ;
    }

}
