package common.data;

import beans.translate.Translate;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.Feature;
import common.BaseClass;
import common.YosException;
import common.data.db.DBConnect;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.*;

public class Row extends HashMap<String, Object> {

    private static final long serialVersionUID = 1L;

    private ArrayList<String> fieldlist;
    private HashMap<String, FieldMeta> fieldMetaMap;

    /**
     * 转换为JsonObject
     *
     * @return
     */
    public JSONObject toJsonObject() {
        return (JSONObject) JSON.toJSON(this);
    }

    public JSONObject toJsonObject(String... fieldnames) {
        JSONObject jsonObject = new JSONObject();
        for (String fieldname : fieldnames) {
            jsonObject.put(fieldname, get(fieldname));
        }
        return jsonObject;
    }


    @Override
    public void putAll(Map<? extends String, ?> m) {
        //排除rowindex字段
        if (this.containsKey("rowindex")) {
            m.remove("rowindex");
        }
        super.putAll(m);
    }

    public long getLong(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Long.parseLong(object.toString());
            }
            if (object instanceof Integer) {
                return Long.valueOf((Integer) object);
            }
            return (Long) object;
        } else {
            return 0L;
        }
    }

    public long getLong(String fieldname, long defaultvalue) {
        return this.containsKey(fieldname) ? getLong(fieldname) : defaultvalue;
    }

    public float getFloat(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Float.parseFloat(object.toString());
            }
            if (object instanceof Integer) {
                return (Float) object;
            }
            if (object instanceof BigDecimal) {
                return ((BigDecimal) object).floatValue();
            }
            if (object instanceof Float) {
                return (Float) object;
            }
            return (Float) object;
        } else {
            return 0F;
        }
    }

    public float getFloat(String fieldname, float defaultvalue) {
        return this.containsKey(fieldname) ? getFloat(fieldname) : defaultvalue;
    }

    public String getString(String fieldname) {
        Object object = get(fieldname);
        if (object == null || String.valueOf(object).equalsIgnoreCase("null")) {
            return "";
        } else {
            return String.valueOf(object);
        }
    }

    public String getString(String fieldname, String defaultvalue) {
        return this.containsKey(fieldname) ? getString(fieldname) : defaultvalue;
    }

    public Integer getInteger(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Integer.parseInt(object.toString());
            }
            if (object instanceof Long) {
                return new Long(object.toString()).intValue();
            }
            if (object instanceof BigDecimal) {
                return ((BigDecimal) object).intValue();
            }
            if (object instanceof Double) {
                return ((Double) object).intValue();
            }
            if (object instanceof Float) {
                return ((Float) object).intValue();
            }
            return (Integer) object;
        } else {
            return 0;
        }
    }

    public Integer getInteger(String fieldname, int defaultvalue) {
        return this.containsKey(fieldname) ? getInteger(fieldname) : defaultvalue;
    }

    public Short getShort(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Short.parseShort(object.toString());
            }
            if (object instanceof Long) {
                return new Long(object.toString()).shortValue();
            }
            if (object instanceof Integer) {
                return new Integer(object.toString()).shortValue();
            }
            if (object instanceof Boolean) {
                return new Integer(((Boolean) object) ? 1 : 0).shortValue();
            }
            return (Short) object;
        } else {
            return 0;
        }
    }

    public Short getShort(String fieldname, short defaultvalue) {
        return this.containsKey(fieldname) ? getShort(fieldname) : defaultvalue;
    }

    public double getDouble(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof String || object instanceof Integer || object instanceof Long || object instanceof Short) {
                return Double.parseDouble(object.toString());
            }
            if (object instanceof Double) {
                return (double) object;
            }
            if (object instanceof BigDecimal) {
                return ((BigDecimal) object).doubleValue();
            }
        }
        return 0d;
    }

    public double getDouble(String fieldname, double defaultvalue) {
        return this.containsKey(fieldname) ? getDouble(fieldname) : defaultvalue;
    }

    public BigDecimal getBigDecimal(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return new BigDecimal(object.toString());
            }
            if (object instanceof Integer) {
                return new BigDecimal((int) object);
            }
            if (object instanceof BigDecimal) {
                return (BigDecimal) object;
            }
        }
        return new BigDecimal("0");
    }

    public BigDecimal getBigDecimal(String fieldname, BigDecimal defaultvalue) {
        return this.containsKey(fieldname) ? getBigDecimal(fieldname) : defaultvalue;
    }

    public Boolean getBoolean(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof Boolean) {
                return (Boolean) object;
            }
            if (object instanceof String) {
                return "true".equalsIgnoreCase(object.toString()) || "1".equals(object.toString());
            }
            if (object instanceof Short) {
                return (Short) object == 1;
            }
            if (object instanceof Integer) {
                return (Integer) object == 1;
            }
            if (object instanceof Long) {
                return (Long) object == 1;
            }
        }
        return false;
    }

    public Boolean getBoolean(String fieldname, Boolean defaultvalue) {
        return this.containsKey(fieldname) ? getBoolean(fieldname) : defaultvalue;
    }

    public Date getDate(String fieldname) {
        Object object = get(fieldname);
        if (object != null) {
            if (object instanceof String) {
                String value = object.toString();
                if (value.length() == 10) {
                    value = value + " 00:00:00.0";
                }
                try {
                    return BaseClass.dateTimeFormat.parse(value);
                } catch (ParseException e) {
                    e.printStackTrace();
                }
            }
            if (object instanceof Date) {
                return (Date) get(fieldname);
            }
            if (object instanceof Timestamp) {
                return new Date(((Timestamp) object).getTime());
            }
        }
        return null;
    }

    public Date getDate(String fieldname, Date defaultvalue) {
        return this.containsKey(fieldname) ? getDate(fieldname) : defaultvalue;
    }

    public JSONArray getJSONArray(String fieldname) {
        Object object = get(fieldname);
        if (object instanceof String && object.toString().equals("")) {
            return new JSONArray();
        }
        try {
            return JSONArray.parseArray(object.toString());
        } catch (Exception e) {
            return new JSONArray();
        }
    }

    public JSONArray getJSONArray(String fieldname, JSONArray defaultvalue) {
        return this.containsKey(fieldname) ? getJSONArray(fieldname) : defaultvalue;
    }


    public JSONObject getJSONObject(String fieldname) {
        Object object = get(fieldname);
        if (object instanceof String && object.toString().equals("")) {
            return new JSONObject();
        }
        try {
            return JSONObject.parseObject(object.toString(), Feature.OrderedField);
        } catch (Exception e) {
            return new JSONObject();
        }
    }

    public JSONObject getJSONObject(String fieldname, JSONObject defaultvalue) {
        return this.containsKey(fieldname) ? getJSONObject(fieldname) : defaultvalue;
    }

    public Rows getRows(String linkname) {
        Object object = get(linkname);
        if (object instanceof Rows) {
            return (Rows) object;
        } else {
            return new Rows();
        }
    }

    public Row getRow(String linkname) {
        Object object = get(linkname);
        if (object instanceof Row) {
            return (Row) object;
        } else {
            return new Row();
        }
    }

    public String toXml() {
        Document document = DocumentHelper.createDocument();
        if (this.size() > 0) {
            Iterator<String> it = this.keySet().iterator();
            ArrayList<String> list = new ArrayList<String>();
            while (it.hasNext()) {
                list.add(it.next());
            }
            Element root = document.addElement("rows");
            Element element = root.addElement("row0");
            for (String key : list) {
                Object value = get(key);
                Element e = element.addElement(key);
                if (value == null) {
                    e.addText("null");
                } else {
                    e.addText(get(key).toString());
                }
            }
        }
        return document.asXML();
    }

    public ArrayList<Object> toList() {
        ArrayList<Object> list = new ArrayList<>();
        for (String fieldname : this.keySet()) {
            list.add(get(fieldname));
        }
        return list;
    }

    public ArrayList<Object> toList(ArrayList<String> fieldnames) {
        ArrayList<Object> list = new ArrayList<>();
        for (String fieldname : fieldnames) {
            list.add(get(fieldname));
        }
        return list;
    }

    public <T> T toPOJO(T pojo) {
        try {
            Field[] pojofields = pojo.getClass().getFields();
            for (Field field : pojofields) {
                String fieldname = field.getName();
                if (this.containsKey(fieldname)) {
                    String fieldtype = field.getType().getSimpleName();
                    switch (fieldtype) {
                        case "String": {
                            field.set(pojo, this.getString(fieldname));
                            break;
                        }
                        case "Date": {
                            field.set(pojo, this.getDate(fieldname));
                            break;
                        }
                        case "Object": {
                            field.set(pojo, this.get(fieldname));
                            break;
                        }
                        case "int":
                        case "Integer": {
                            field.set(pojo, this.getInteger(fieldname));
                            break;
                        }
                        case "long":
                        case "Long": {
                            field.set(pojo, this.getLong(fieldname));
                            break;
                        }
                        case "short":
                        case "Short": {
                            field.set(pojo, this.getShort(fieldname));
                            break;
                        }
                        case "double":
                        case "Double": {
                            field.set(pojo, this.getBigDecimal(fieldname));
                            break;
                        }
                        case "boolean":
                        case "Boolean": {
                            field.set(pojo, this.getBoolean(fieldname));
                            break;
                        }
                        default:
                            break;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pojo;
    }


    public void setFieldList(ArrayList<String> fieldlist) {
        this.fieldlist = fieldlist;
    }

    /**
     * 获取Rows所有的字段
     *
     * @return
     */
    public ArrayList<String> getFieldList() {
        return this.fieldlist;
    }

    public HashMap<String, FieldMeta> getFieldMetaMap() {
        return fieldMetaMap;
    }

    public FieldMeta getFieldMeta(String fieldname) {
        return fieldMetaMap.get(fieldname);
    }

    public ArrayList<String> getTableNames() {
        ArrayList<String> arrayList = new ArrayList<>();
        for (FieldMeta fieldMeta : fieldMetaMap.values()) {
            if (!arrayList.contains(fieldMeta.getTable_name())) {
                arrayList.add(fieldMeta.getTable_name());
            }
        }
        return arrayList;
    }

    public void setFieldMetaMap(HashMap<String, FieldMeta> fieldMetaMap) {
        this.fieldMetaMap = fieldMetaMap;
    }

    @Deprecated
    public void setFieldTypeMap(HashMap<String, Class> keytypemap) {
        if (this.fieldMetaMap == null) {
            this.fieldMetaMap = new HashMap<>();
        }
        for (String fieldname : keytypemap.keySet()) {
            FieldMeta fieldMeta = null;
            if (this.fieldMetaMap.containsKey(fieldname)) {
                fieldMeta = this.fieldMetaMap.get(fieldname);
            } else {
                fieldMeta = new FieldMeta();
            }
            fieldMeta.setFieldtype(keytypemap.get(fieldname));
            this.fieldMetaMap.put(fieldname, fieldMeta);
        }
    }

    /**
     * 对指定对字段进行简单的内容匹配翻译
     *
     * @param languagecode
     * @param fieldnames
     * @throws YosException
     */
    public void translate_simple(String languagecode, String... fieldnames) throws YosException {
        for (String fieldname : fieldnames) {
            put(fieldname, Translate.getSimpleTranslate(languagecode, getString(fieldname)));
        }
    }

    /**
     * 数据精准翻译，如果没有匹配到精准翻译，则再匹配简单翻译
     *
     * @param languagecode
     * @param idcolumn
     * @param fieldnames
     * @throws YosException
     */
    public void translate(String languagecode, String idcolumn, String... fieldnames) throws YosException {
        for (String fieldname : fieldnames) {
            FieldMeta fieldMeta = getFieldMeta(fieldname);
            String tablename = fieldMeta.getTable_name();
            if (tablename.equals("")) {
                continue;
            }
            put(fieldname, Translate.getTranslate(languagecode, tablename, getLong(idcolumn), fieldMeta.getColumn_name(), getString(fieldname)));
        }
    }


    /**
     * 将当前row转换成一个新增数据
     *
     * @param tablename
     * @return
     * @throws YosException
     */
    public InsertSQL getInsertSQL(String tablename) throws YosException {
        InsertSQL insertSQL = SQLFactory.createInsertSQL(new DBConnect(), tablename);
        ArrayList<String> tc = insertSQL.getTableColumnNames(tablename);
        for (String fieldname : this.keySet()) {
            if (tc.contains(fieldname)) {
                String value = getString(fieldname);
                value = value.equals("") ? "null" : value;
                insertSQL.setValue(fieldname, value);
            }
        }
        return insertSQL;
    }

    /**
     * 将当前row转换成一个更新数据
     *
     * @param tablename
     * @return
     * @throws YosException
     */
    public UpdateSQL getUpdateSQL(String tablename) throws YosException {
        UpdateSQL updateSQL = SQLFactory.createUpdateSQL(new DBConnect(), tablename);
        if (!this.containsKey(updateSQL.uniquecolumnname)) {
            throw new YosException("当前row没有包含唯一ID字段");
        }
        ArrayList<String> tc = updateSQL.getTableColumnNames(tablename);
        for (String fieldname : this.keySet()) {
            if (fieldname.equalsIgnoreCase(updateSQL.uniquecolumnname)) {
                updateSQL.setUniqueid(getLong(fieldname));
            } else if (fieldname.equalsIgnoreCase("siteid")) {
                updateSQL.setSiteid(getString(fieldname));
            } else if (tc.contains(fieldname)) {
                String value = getString(fieldname);
                value = value.equals("") ? "null" : value;
                updateSQL.setValue(fieldname, value);
            }
        }
        return updateSQL;
    }

    String toStringvalue = "";

    public Row setToStringValue(String... fieldnames) {
        for (String fieldname : fieldnames) {
            toStringvalue = toStringvalue + " " + getString(fieldname);
        }
        return this;
    }

    @Override
    public String toString() {
        if (toStringvalue.equals("")) {
            return super.toString();
        } else {
            return toStringvalue;
        }
    }
}
