package utility.mysql;

import beans.parameter.Parameter;
import common.YosException;
import common.data.Row;
import common.data.Rows;
import common.data.db.DBConnect;
import common.data.db.DBConnectPool;
import utility.zip.ZipUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.sql.*;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;

public class MysqlExport {

    private Statement stmt;
    private String database;
    private String tablename = "";
    private String datakey;

    /**
     * 备份整个数据库
     */
    public MysqlExport() throws YosException {
        this.datakey = "default";
        this.database = getDBname();
    }

    public MysqlExport(String datakey) throws YosException {
        this.datakey = datakey;
        this.database = getDBname();
    }
//
//    /**
//     * 备份指定的数据表
//     *
//     * @param tablename
//     */
//    public MysqlExport(String tablename) {
//        database = getDBname();
//        this.tablename = tablename;
//    }

    /**
     * 导出入口
     *
     * @throws Exception exception
     */
    public boolean export() {
        try {
            if (Parameter.get("system_db_backupswitch").equalsIgnoreCase("true") || Parameter.get("system_db_backupswitch").equalsIgnoreCase("1")) {
                DBConnectPool dbConnectPool = new DBConnectPool();
                DBConnectPool.YosConnection connection = dbConnectPool.getConnect(datakey);
                stmt = connection.getConnection().createStatement(ResultSet.TYPE_SCROLL_INSENSITIVE, ResultSet.CONCUR_READ_ONLY);
                String path = Parameter.get("system_db_backuppath");
                File file = new File(path);
                if (!file.exists() && !file.mkdir()) {
                    throw new IOException("创建备份文件路径失败 " + file.getAbsolutePath());
                }
                String filepath = path + "/" + getSqlFilename();
                OutputStreamWriter oStreamWriter = new OutputStreamWriter(new FileOutputStream(filepath), StandardCharsets.UTF_8);
                exportToSql(oStreamWriter);
                oStreamWriter.close();
                dbConnectPool.close(connection);

                File sqlfile = new File(filepath);
                if (sqlfile.exists()) {
                    ZipUtil.zip(new File[]{sqlfile}, filepath.replace(".sql", ".zip"));
                    sqlfile.delete();
                }
                System.gc();
            }
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
        return true;
    }

    /**
     * 导出所有表的结构和数据
     *
     * @return String
     * @throws SQLException exception
     */
    private void exportToSql(OutputStreamWriter sqlWriter) throws SQLException, IOException, YosException {
        sqlWriter.append("/*");
        sqlWriter.append("\n Navicat Premium Data Transfer");
        sqlWriter.append("\n");
        sqlWriter.append("\n Source Server         : yos");
        sqlWriter.append("\n Source Server Type    : MySQL");
        sqlWriter.append("\n Source Server Version : 80028");
        sqlWriter.append("\n Source Host           : localhost:3306");
        sqlWriter.append("\n Source Schema         : ").append(this.database);
        sqlWriter.append("\n");
        sqlWriter.append("\n Target Server Type    : MySQL");
        sqlWriter.append("\n Target Server Version : 80028");
        sqlWriter.append("\n File Encoding         : 65001");
        sqlWriter.append("\n");
        sqlWriter.append("\n Date: 29/03/2023 08:43:12");
        sqlWriter.append("\n*/");
        sqlWriter.append("\n");
        sqlWriter.append("\nSET NAMES utf8mb4;");
        sqlWriter.append("\nSET FOREIGN_KEY_CHECKS = 0;");

        //获取数据库所有表名
        List<String> tables = getAllTables(database, stmt);

        //遍历表
        for (String table : tables) {
            try {
                table = table.trim();
                //生成表格创建语句
                sqlWriter.append(getTableInsertStatement(table));
                //生成表数据插入语句
                getDataInsertStatement(sqlWriter, table);
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        /**
         * 生成视图
         */
        sqlWriter.append(getViewInsertStatement(stmt));
        /**
         * 生成触发器
         */
        sqlWriter.append(getTriggersInsertStatement(stmt));
        sqlWriter.append("\n\nSET FOREIGN_KEY_CHECKS = 1;\n");
    }

    /**
     * 生成create语句
     *
     * @param table the table concerned
     * @return String
     * @throws SQLException exception
     */
    private String getTableInsertStatement(String table) throws SQLException {
        StringBuilder sql = new StringBuilder();
        ResultSet rs;

        if (table != null && !table.isEmpty()) {
            rs = stmt.executeQuery("SHOW CREATE TABLE " + "`" + table + "`;");
            while (rs.next()) {
                String tablename = rs.getString(1);
                String createtablesql = rs.getString(2);
                sql.append("\n\n-- ----------------------------");
                sql.append("\n").append("-- Table structure for ").append(tablename);
                sql.append("\n-- ----------------------------");
                sql.append("\nDROP TABLE IF EXISTS `").append(tablename).append("`;\n");
                sql.append(createtablesql).append(" ROW_FORMAT = Dynamic;");
            }
        }
        return sql.toString();
    }

    /**
     * 生成insert语句
     *
     * @param table the table to get inserts statement for
     * @return String generated SQL insert
     * @throws SQLException exception
     */
    private void getDataInsertStatement(OutputStreamWriter sqlWriter, String table) throws SQLException, IOException {
        ResultSet rs = stmt.executeQuery("SELECT * FROM " + "`" + table + "`;");
        rs.last();
        int rowCount = rs.getRow();
        if (rowCount <= 0) {
            return;
        }
        sqlWriter.append("\n\n-- ----------------------------").append("\n-- Records of ").append(table).append("\n-- ----------------------------\n");

        ResultSetMetaData metaData = rs.getMetaData();
        int columnCount = metaData.getColumnCount();

        rs.beforeFirst();
        while (rs.next()) {
            StringBuilder insertsql = new StringBuilder();
            insertsql.append("INSERT INTO `").append(table).append("` VALUES");
            insertsql.append("(");
            for (int i = 0; i < columnCount; i++) {
                int columnType = metaData.getColumnType(i + 1);
                int columnIndex = i + 1;
                if (Objects.isNull(rs.getObject(columnIndex))) {
                    insertsql.append("NULL").append(", ");
                } else if (columnType == Types.BIGINT) {
                    insertsql.append(rs.getLong(columnIndex)).append(", ");
                } else if (columnType == Types.INTEGER || columnType == Types.TINYINT || columnType == Types.BIT || columnType == Types.SMALLINT) {
                    insertsql.append(rs.getInt(columnIndex)).append(", ");
                } else {
                    String val = rs.getString(columnIndex);
                    val = val.replace("'", "\\'").replace("\n", "\\n").replace("\r", "\\r");
                    insertsql.append("'").append(val).append("', ");
                }
            }
            insertsql.deleteCharAt(insertsql.length() - 1).deleteCharAt(insertsql.length() - 1);
            if (rs.isLast()) {
                insertsql.append(")");
            } else {
                insertsql.append(");\n");
            }
            sqlWriter.append(insertsql.toString());
        }
        sqlWriter.append(";");
    }

    /**
     * 生成视图语句
     *
     * @return String generated SQL insert
     * @throws SQLException exception
     */
    private String getViewInsertStatement(Statement stmt) throws SQLException {
        StringBuilder sql = new StringBuilder();
        List<String> viewList = getAllViews(database, stmt);
        for (String viewname : viewList) {
            sql.append("\n\n-- ----------------------------");
            sql.append("\n-- View structure for ").append(viewname);
            sql.append("\n-- ----------------------------");
            sql.append("\nDROP VIEW IF EXISTS `").append(viewname).append("`;");
            try {
                Rows rows = new DBConnect().runSqlQuery("SHOW CREATE VIEW " + viewname + ";");
                for (Row row : rows) {
                    String viewsql = row.getString("Create View");
                    viewsql.replace("DEFINER=`root`@`localhost` ", "");
                    sql.append("\n").append(viewsql);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            sql.append(";");
        }
        return sql.toString();
    }

    /**
     * 生成触发器语句
     *
     * @return String generated SQL insert
     * @throws SQLException exception
     */
    private String getTriggersInsertStatement(Statement stmt) throws SQLException {
        StringBuilder sql = new StringBuilder();
        ResultSet rs = stmt.executeQuery("SHOW TRIGGERS;");
        while (rs.next()) {
            String triggername = rs.getString("Trigger");
            String table = rs.getString("Table");
            sql.append("\n\n-- ----------------------------");
            sql.append("\n-- Triggers structure for table ").append(table);
            sql.append("\n-- ----------------------------");
            sql.append("\nDROP TRIGGER IF EXISTS `").append(triggername).append("`;");
            sql.append("\ndelimiter ;;");
            try {
                Rows rows = new DBConnect().runSqlQuery("SHOW CREATE TRIGGER " + triggername + ";");
                for (Row row : rows) {
                    String triggersql = row.getString("SQL Original Statement");
                    triggersql.replace("DEFINER=`root`@`localhost` ", "");
                    sql.append("\n").append(triggersql);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            sql.append("\n;;");
            sql.append("\ndelimiter ;;");
        }
        return sql.toString();
    }

    /**
     * 生成文件名
     * sql file name.
     *
     * @return String
     */
    public String getSqlFilename() {
        if (tablename.equals("")) {
            return new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + "_" + database + ".sql";
        } else {
            return new SimpleDateFormat("yyyyMMddHHmmss").format(new Date()) + "_" + database + "." + tablename + ".sql";
        }
    }

    /**
     * 获取所有表
     *
     * @param database the database name
     * @param stmt     Statement object
     * @return List\<String\>
     * @throws SQLException exception
     */
    private List<String> getAllTables(String database, Statement stmt) throws SQLException {
        List<String> table = new ArrayList<>();
        ResultSet rs;
        if (tablename.equals("")) {
            rs = stmt.executeQuery("SHOW TABLE STATUS FROM `" + database + "`;");
        } else {
            rs = stmt.executeQuery("SHOW TABLE STATUS FROM `" + database + "` where Name='" + tablename + "';");
        }
        while (rs.next()) {
            table.add(rs.getString("Name"));
        }
        for (String view : getAllViews(database, stmt)) {
            table.remove(view);
        }
        return table;
    }

    private List<String> getAllViews(String database, Statement stmt) throws SQLException {
        List<String> view = new ArrayList<>();
        ResultSet rs = stmt.executeQuery("SHOW FULL TABLES from `" + database + "` WHERE TABLE_TYPE='VIEW';");
        while (rs.next()) {
            view.add(rs.getString("Tables_in_" + database));
        }
        return view;
    }

    /**
     * 获取数据库名称
     *
     * @return
     */
    private String getDBname() throws YosException {
        return new DBConnect(datakey).getDBName();
    }
}
