package utility.huawei.obs;

import beans.parameter.Parameter;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.obs.services.internal.utils.ServiceUtils;
import com.obs.services.model.*;
import com.obs.services.model.fs.DropFolderRequest;
import common.YosException;
import org.apache.commons.io.IOUtils;
import utility.huawei.obs.mpc.MediaMetaData;
import utility.huawei.obs.mpc.MpcFactory;
import utility.tools.WebRequest;
import utility.zip.ZipUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

public class BucketFile extends Bucket {

    public BucketFile(String siteid) throws YosException {
        super(siteid);
    }

    /**
     * 流文件上传，如果文件类型是图片,则会自动生成缩略图
     */
    public void upload(String filename, InputStream inputStream, boolean iscreateMinImage) {
        obsClient.putObject(bucketname, filename, inputStream);
        /**
         * 如果是图片文件，则同时生成一个缩略图文件
         */
        if (iscreateMinImage) {
            getMinImageUrl(filename);
            getCompressImageUrl(filename);
        }
    }

    /**
     * 流文件上传，如果文件类型是图片,则会自动生成缩略图
     */
    public void upload(String filename, File file) {
        obsClient.putObject(bucketname, filename, file);
    }


    /**
     * 文件删除,如果存在缩略文件，则缩略文件会一起删除
     *
     * @param objectname
     */
    public void deleteFile(String objectname) {
        obsClient.deleteObject(bucketname, objectname);
    }

    public void deleteFolder(String foldername) {
        DropFolderRequest dropFolderRequest = new DropFolderRequest(bucketname, foldername);
        obsClient.dropFolder(dropFolderRequest);
    }

    public boolean doesObjectExist(String objectname) {
        return obsClient.doesObjectExist(bucketname, objectname);
    }

    /**
     * 获取文件下载地址
     */
    public String getFileUrl(String filename) {
        //临时访问地址有效时间
        //long expireSeconds = 3600L;
        // 创建ObsClient实例
        TemporarySignatureRequest request = new TemporarySignatureRequest(HttpMethodEnum.GET, 3600);
        request.setBucketName(bucketname);
        request.setObjectKey(filename);
        // 生成临时授权URL
        TemporarySignatureResponse response = obsClient.createTemporarySignature(request);

        //截取永久访问url，需要桶设置公共访问策略
        String url = response.getSignedUrl();
        return url.substring(0, url.indexOf("?"));
    }


    /**
     * 获取文件缩略图下载地址,缩略图的名称为在源文件的前面加上"s_"
     */
    public String getMinImageUrl(String filename) {
        /*
         * 缩略图名称
         */
        String filename_short = "s_" + filename;
        if (!obsClient.doesObjectExist(bucketname, filename_short)) {
            //临时访问地址有效时间
            // 创建ObsClient实例
            TemporarySignatureRequest request = new TemporarySignatureRequest(HttpMethodEnum.GET, 3600);
            request.setBucketName(bucketname);
            request.setObjectKey(filename);

            // 设置图片处理参数，对图片依次进行缩放、旋转
            Map<String, Object> queryParams = new HashMap<String, Object>();

            boolean istohight = false;
            try {
                MediaMetaData mediaMetaData = new MpcFactory().getMetaData(siteid, filename);
                if (mediaMetaData.getWidth() > 0 && mediaMetaData.getHeight() / mediaMetaData.getWidth() > 3) {
                    //当图片高度太高时，调整图片当缩略样式
                    istohight = true;
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            if (istohight) {
                queryParams.put("x-image-process", "image/resize,m_fill,h_300,w_300");
            } else {
                queryParams.put("x-image-process", "image/resize,m_lfit,h_150");
            }

            try {
                queryParams.put("x-image-save-object", ServiceUtils.toBase64(filename_short.getBytes(StandardCharsets.UTF_8)));
            } catch (Exception e) {
                e.printStackTrace();
            }
            request.setQueryParams(queryParams);
            // 生成临时授权URL
            TemporarySignatureResponse response = obsClient.createTemporarySignature(request);
            //临时访问，生成缩略图（华为云在第一次访问该地址时才会生成实体文件）
            new WebRequest().doGet_simple(response.getSignedUrl());
        }
        return getFileUrl(filename_short);
    }

    /**
     * 获取文件缩略图下载地址,缩略图的名称为在源文件的前面加上"c_"
     */
    public String getCompressImageUrl(String filename) {
        /**
         * 缩略图名称
         */
        String filename_short = "c_" + filename;
        if (!obsClient.doesObjectExist(bucketname, filename_short)) {
            //临时访问地址有效时间
            // 创建ObsClient实例
            TemporarySignatureRequest request = new TemporarySignatureRequest(HttpMethodEnum.GET, 3600);
            request.setBucketName(bucketname);
            request.setObjectKey(filename);

            // 设置图片处理参数，对图片依次进行缩放、旋转
            Map<String, Object> queryParams = new HashMap<String, Object>();
            try {
                MpcFactory mpcFactory = new MpcFactory();
                MediaMetaData data = mpcFactory.getMetaData(siteid, filename);
                if (data.getWidth() > 5000 || data.getHeight() > 5000) {
                    queryParams.put("x-image-process", "image/resize,p_30");//长宽均压缩30%
                } else {
                    queryParams.put("x-image-process", "image/resize,p_50");//长宽均压缩50%
                }
                queryParams.put("x-image-save-object", ServiceUtils.toBase64(filename_short.getBytes(StandardCharsets.UTF_8)));
            } catch (Exception e) {
                e.printStackTrace();
            }
            request.setQueryParams(queryParams);
            // 生成临时授权URL
            TemporarySignatureResponse response = obsClient.createTemporarySignature(request);
            //临时访问，生成缩略图（华为云在第一次访问该地址时才会生成实体文件）
            new WebRequest().doGet_simple(response.getSignedUrl());
        }
        return getFileUrl(filename_short);
    }


    public ObjectMetadata getObjectMetadata(String objectname) {
        return obsClient.getObjectMetadata(bucketname, objectname);
    }

    public InputStream getObjectInputStream(String objectname) {
        ObsObject obsObject = obsClient.getObject(bucketname, objectname);
        return obsObject.getObjectContent();
    }

    /**
     * 关闭
     */
    public void close() {
        if (obsClient != null) {
            try {
                obsClient.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public String getUploadUrl(String contentType, String objectname, HttpMethodEnum type) {
        long expireSeconds = 3600L;
        Map<String, String> headers = new HashMap<String, String>();
        headers.put("Content-Type", contentType);
        TemporarySignatureRequest request = new TemporarySignatureRequest(type, expireSeconds);
        request.setBucketName(bucketname);
        request.setObjectKey(objectname);
        request.setHeaders(headers);

        TemporarySignatureResponse response = obsClient.createTemporarySignature(request);
        return response.getSignedUrl();
    }

    public enum FileType {
        pdf, svg, other
    }

    public String getUploadUrl(FileType filetype, String objectname, HttpMethodEnum type) {
        String contentType = "application/octet-stream";
        if (type.equals(HttpMethodEnum.POST)) {
            contentType = "multipart/form-data";
        }
        switch (filetype) {
            case pdf: {
                contentType = "application/pdf";
                break;
            }
            case svg: {
                contentType = "image/svg+xml";
                break;
            }
        }
        return getUploadUrl(contentType, objectname, type);
    }

    /**
     * 文件打包并返回文件地址
     *
     * @param packpath   "cacheFiles/13132131313/文件打包测试.zip"
     * @param filesArray [
     *                   {
     *                   "filepath": "202104141618363277742B784f1f19.jpg",
     *                   "packpath":"图片1.jpg"
     *                   }
     *                   ]
     * @return
     */
    public String getPackFileUrl(String packpath, JSONArray filesArray) throws YosException {
        JSONObject requestObject = new JSONObject();
        {
            JSONObject parameterObject = new JSONObject();
            parameterObject.put("endPoint", Parameter.get("attachment_huawei_obs_endpoint"));
            parameterObject.put("ak", Parameter.get("attachment_huawei_obs_ak"));
            parameterObject.put("sk", Parameter.get("attachment_huawei_obs_sk"));
            parameterObject.put("bucketName", bucketname);
            requestObject.put("parameter", parameterObject);
        }
        requestObject.put("packfilepath", packpath);
        requestObject.put("files", filesArray);

        WebRequest webRequest = new WebRequest();
        String result = webRequest.doPost(requestObject.toJSONString(), "https://2a1c33bfc16c45d2a6f6889a70ffc66f.apig.cn-east-2.huaweicloudapis.com/obsfilepacking");
        logger.info("文件夹打包下载" + result);
        if (doesObjectExist(packpath)) {
            return getFileUrl(packpath);
        }
        return null;
    }
}
