package common.data;

import com.alibaba.fastjson.JSONObject;
import common.Controller;
import common.YosException;
import common.data.db.DBConnect;

import java.util.*;

public class UpdateSQL extends BaseSQL<UpdateSQL> {
    /**
     * SQL参数Map
     */
    private HashMap<String, Object> keyvaluemap;

    protected UpdateSQL(Controller controller, String tablename) throws YosException {
        super(controller, tablename);
    }

    protected UpdateSQL(DBConnect dbConnect, String tablename) throws YosException {
        super(dbConnect, tablename);
    }

    /**
     * 根据row快速添加更新值
     *
     * @param row
     * @return
     * @throws YosException
     */
    public UpdateSQL addRowValue(Row row) throws YosException {
        ArrayList<String> fieldlist = tableColumnNames.get(this.tablename);
        for (String fieldname : row.keySet()) {
            if (fieldlist.contains(fieldname)) {
                if (fieldname.equalsIgnoreCase(this.uniquecolumnname)) {
                    setUniqueid(row.getLong(fieldname));
                } else if (fieldname.equalsIgnoreCase("siteid")) {
                    setSiteid(row.getString(fieldname));
                } else {
                    setValue(fieldname, row.get(fieldname));
                }
            }
        }
        return this;
    }

    /**
     * 根据JSONObject快速添加更新值
     *
     * @param object
     * @return
     * @throws YosException
     */
    public UpdateSQL addJSONValue(JSONObject object) throws YosException {
        ArrayList<String> fieldlist = tableColumnNames.get(this.tablename);
        for (String fieldname : object.keySet()) {
            if (fieldlist.contains(fieldname)) {
                if (fieldname.equalsIgnoreCase(this.uniquecolumnname)) {
                    setUniqueid(object.getLong(fieldname));
                } else if (fieldname.equalsIgnoreCase("siteid")) {
                    setSiteid(object.getString(fieldname));
                } else {
                    setValue(fieldname, object.get(fieldname));
                }
            }
        }
        return this;
    }

    /**
     * 设置主表别名
     *
     * @param table_alias
     * @return
     */
    public UpdateSQL setTableAlias(String table_alias) {
        this.table_alias = table_alias;
        return this;
    }

    /**
     * 如果通过数据id进行更新，则可能需要进行版本号的判断。
     *
     * @param uniqueid
     * @return
     */
    @Override
    public UpdateSQL setUniqueid(long uniqueid) throws YosException {
        return super.setUniqueid(uniqueid);
    }

    /**
     * 如果通过数据id进行更新
     *
     * @param uniqueid 数据ID
     * @param version  数据版本号
     * @return
     */
    public UpdateSQL setUniqueid(long uniqueid, int version) throws YosException {
        setUniqueid(uniqueid);
        setDataVersion(version);
        return this;
    }

    /**
     * 设置更新版本号
     *
     * @param version
     * @return
     */
    public UpdateSQL setDataVersion(int version) {
        if (this.tableColumnNames.get(tablename).contains(getdataversioncolumnname(tablename))) {
            setValue(getdataversioncolumnname(tablename), version);
        }
        return this;
    }

    /**
     * 为SQL实例添加文本类型参数值
     *
     * @param key
     * @param value
     */
    public UpdateSQL setValue(String key, Object value) {
        if (this.keyvaluemap == null) {
            this.keyvaluemap = new HashMap<>();
        }
        if (value instanceof Boolean) {
            value = (Boolean) value ? 1 : 0;
        }
        this.keyvaluemap.put(key, value);
        return this;
    }

    /**
     * 为SQL实例添加参数，参数值会自动从controller.content中获取
     *
     * @param key
     * @return
     */
    public UpdateSQL setValue(String key) {
        if (this.controller != null && this.controller.content.containsKey(key)) {
            setValue(key, this.controller.content.get(key));
        }
        return this;
    }

    /**
     * 为SQL实例添加int类型的参数值，
     *
     * @param key
     */
    public UpdateSQL setDateValue(String key) {
        if (this.keyvaluemap == null) {
            this.keyvaluemap = new HashMap<>();
        }
        this.keyvaluemap.put(key, Date.class);
        return this;
    }

    /**
     * 数据复制
     *
     * @param sourceid      源数据ID
     * @param excludeFields 需要排除的字段
     * @return
     */
    public UpdateSQL copy(long sourceid, String... excludeFields) throws YosException {
        QuerySQL querySQL = new QuerySQL(dbConnect, tablename);
        querySQL.setUniqueid(sourceid);
        Rows rows = querySQL.query();
        if (rows.isNotEmpty()) {
            ArrayList<String> excludeFieldList = new ArrayList<>(Arrays.asList(excludeFields));
            excludeFieldList.add(uniquecolumnname);
            excludeFieldList.add(getuniquecolumnname(tablename));
            excludeFieldList.add("createby");
            excludeFieldList.add("createdate");
            excludeFieldList.add("createuserid");
            excludeFieldList.add("changeby");
            excludeFieldList.add("changedate");
            excludeFieldList.add("changeuserid");
            excludeFieldList.add("rowindex");
            excludeFieldList.add("createdepid");
            excludeFieldList.add("authdepid");
            excludeFieldList.add("authuserid");
            excludeFieldList.add(getdataversioncolumnname(tablename));
            Row row = rows.getRow(0);
            for (String fieldname : row.keySet()) {
                if (!excludeFieldList.contains(fieldname)) {
                    this.setValue(fieldname, row.get(fieldname));
                }
            }
        }
        return this;
    }

    /**
     * 异表数据复制
     *
     * @param sourceobject 数据源表
     * @param sourceid     源数据ID
     * @param copyfields   需要复制的字段，如两个表字段不一致，则可通过特殊符号进行标注如："objectname:tablename",冒号前为来源字段，后为当前表字段
     * @return
     */
    public UpdateSQL copy(String sourceobject, long sourceid, String... copyfields) throws YosException {
        QuerySQL querySQL = new QuerySQL(dbConnect, sourceobject);
        querySQL.setUniqueid(sourceid);
        Rows sourceRows = querySQL.query();
        if (sourceRows.isNotEmpty()) {
            Row sourceRow = sourceRows.getRow(0);
            Set<String> rowketSet = sourceRow.keySet();
            for (String fieldname : copyfields) {
                if (fieldname.contains(":")) {
                    String[] fs = fieldname.split(":");
                    this.setValue(fs[1], sourceRow.get(fs[0]));
                } else if (rowketSet.contains(fieldname)) {
                    this.setValue(fieldname, sourceRow.get(fieldname));
                }
            }
        }
        return this;
    }

    public String getSQL() throws YosException {
        setDefaultField();

        StringBuilder updateSql = new StringBuilder("update ").append(tablename).append(" set \r\n");
        boolean isfirstfield = true;
        for (String fieldname : this.keyvaluemap.keySet()) {
            Object value = this.keyvaluemap.get(fieldname);
            if (value == null || "null".equalsIgnoreCase(value.toString())) {
                updateSql.append(isfirstfield ? "" : ",\r\n").append(fieldname).append("=").append("NULL");
            } else {
                if (value == Date.class) {
                    updateSql.append(isfirstfield ? "" : ",\r\n").append(fieldname).append("=now()");
                } else {
                    if (value instanceof Date || value instanceof Calendar) {
                        Date date = value instanceof Date ? (Date) value : ((Calendar) value).getTime();
                        if ("date".equals(getColumnType(tablename, fieldname))) {
                            updateSql.append(isfirstfield ? "" : ",\r\n").append(fieldname).append("='").append(getDate_Str(date)).append("'");
                        } else {
                            updateSql.append(isfirstfield ? "" : ",\r\n").append(fieldname).append("='").append(getDateTime_Str(date)).append("'");
                        }
                    } else {
                        updateSql.append(isfirstfield ? "" : ",\r\n").append(fieldname).append("='").append(value).append("'");
                    }
                }
            }
            isfirstfield = false;
        }
        if (!wherelist.isEmpty()) {
            updateSql.append("\r\nwhere ");
            boolean isfirstwhere = true;
            for (String where : wherelist) {
                updateSql.append(isfirstwhere ? "(" : " and (").append(parameterDo(where)).append(")");
                isfirstwhere = false;
            }
        }
        return updateSql.toString();
    }

    private void setDefaultField() throws YosException {
        ArrayList<String> list = getTableColumnNames(tablename);
        if (controller != null) {
            if (list.contains("changeuserid") && !keyvaluemap.containsKey("changeuserid")) {
                this.setValue("changeuserid", controller.userid);
            }
            if (list.contains("changeby") && !keyvaluemap.containsKey("changeby")) {
                this.setValue("changeby", controller.username);
            }
        }
        if (list.contains("changedate") && !keyvaluemap.containsKey("changedate")) {
            this.setDateValue("changedate");
        }
    }

    private int effectrows = -1;//影响行数

    /**
     * 执行更新操作
     */
    public int update() throws YosException {
        if (effectrows == -1) {
            effectrows = dbConnect.runSqlUpdate(getSQL());
        }
        return effectrows;
    }

    private Rows rows = null;

    /**
     * 获取更新后的行，如果执行此方法前没有执行过更新操作，则会先执行更新操作
     *
     * @return
     * @throws YosException
     */
    public Rows getRow() throws YosException {
        if (rows == null) {
            if (effectrows == -1) {
                update();
            }
            if (effectrows > 0) {
                StringBuilder querySql = new StringBuilder("select * from ").append(tablename).append(" where ( 1=1 ) ");
                for (String where : wherelist) {
                    querySql.append(" and (").append(parameterDo(where)).append(")");
                }
                rows = dbConnect.runSqlQuery(querySql.toString());
            } else {
                rows = new Rows();
            }
        }
        return rows;
    }
}
