/**
 *
 */
package common.data.db;

import com.alibaba.fastjson.JSON;
import com.mysql.cj.result.Field;
import common.BaseClass;
import common.YosException;
import common.YosLogger;
import common.data.FieldMeta;
import common.data.Row;
import common.data.Rows;
import common.data.SQLFactory;

import java.math.BigDecimal;
import java.sql.*;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;

/**
 * @author 沈静伟
 *
 * 获取一个数据库连接
 */
public class DBConnect implements YosLogger {

    private boolean createsqllog = false;
    /**
     * 数据库连接名称
     */
    final public String databasekey;
    private ArrayList<SQLLog> sqlLogList = new ArrayList<>();

    /**
     * 默认构造函数，databasekey=default 表示此次获取yos系统数据库连接
     */
    public DBConnect() {
        this.databasekey = "default";
    }

    /**
     * 获取指定的数据库连接，databasekey需要在yos中注册
     * @param databasekey
     */
    public DBConnect(String databasekey) {
        this.databasekey = databasekey;
    }

    /**
     * 获取数据库名称
     * @return
     */
    public String getDBName() throws YosException {
        if (this.databasekey.equals("default")) {
            return runSqlQuery(0, "select database() as dbname").getString("dbname");
        } else {
            return runSqlQuery(0, "select dbname from sys_datasource where datakey='" + databasekey + "'").getString("dbname");
        }
    }

    public String getDBProduct() throws YosException {
        String name = "";
        DBConnectPool.YosConnection yosConnection = new DBConnectPool().getConnect(databasekey);
        try {
            name = yosConnection.getConnection().getMetaData().getDatabaseProductName();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            new DBConnectPool().close(yosConnection);
        }
        return name;
    }

    /**
     * 是否需要记录sql日志
     * @param createsqllog
     */
    public void setCreatesqllog(boolean createsqllog) {
        this.createsqllog = createsqllog;
    }

    /**
     * 获取SQL执行日志
     * @return
     */
    public ArrayList<SQLLog> getSqlLogList() {
        return sqlLogList;
    }

    public void clearSqlLogList() {
        sqlLogList.clear();
    }

    /**
     * 执行sql查询语句，并获取指定行的数据
     * @param index
     * @param sqlFactory
     * @return
     */
    public Row runSqlQuery(int index, SQLFactory sqlFactory) throws YosException {
        return runSqlQuery(sqlFactory).get(index);
    }

    /**
     * 执行sql查询语句，并获取指定行的数据
     * @param index
     * @param SQL
     * @return
     */
    public Row runSqlQuery(int index, String SQL) throws YosException {
        return runSqlQuery(SQL).get(index);
    }

    /**
     * 执行sql查询语句
     * @param sqlFactory
     * @return
     */
    public Rows runSqlQuery(SQLFactory sqlFactory) throws YosException {
        return runSqlQuery(sqlFactory.getSQL(this));
    }

    public static HashMap<String, String> dbprodnamemap = new HashMap<>();

    private String getDBProdName(Connection conn) {
        try {
            if (!dbprodnamemap.containsKey(databasekey)) {
                dbprodnamemap.put(databasekey, conn.getMetaData().getDatabaseProductName());
            }
            return dbprodnamemap.get(databasekey);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * 执行sql查询语句
     * @param SQL
     * @return
     */
    public Rows runSqlQuery(String SQL) throws YosException {
        long starttimes = Calendar.getInstance().getTimeInMillis();
        DBConnectPool.YosConnection yosConnection = new DBConnectPool().getConnect(databasekey);
        Connection conn = yosConnection.getConnection();
        if (conn == null) {
            if (createsqllog) {
                sqlLogList.add(new SQLLog(SQL, databasekey + "数据库连接失败"));
            }
            throw new YosException(databasekey + "数据库连接失败");
        }
        String DBProdName = getDBProdName(conn);
        Rows rows = new Rows();
        ResultSet resultSet = null;
        ResultSet found_rowsResultSet = null;
        PreparedStatement statement = null;
        long total = -1;
        long pageTotal = -1;
        try {
            statement = conn.prepareStatement(SQL);
            resultSet = statement.executeQuery();
            ResultSetMetaData rsmd = resultSet.getMetaData();
            int colCount = rsmd.getColumnCount();
            if (DBProdName.equals("MySQL") && SQL.contains("SQL_CALC_FOUND_ROWS")) {
                statement = conn.prepareStatement("SELECT FOUND_ROWS()");
                found_rowsResultSet = statement.executeQuery();
                while (found_rowsResultSet.next()) {
                    total = found_rowsResultSet.getLong(1);
                }
            }

            ArrayList<String> colNameList = new ArrayList<String>();
            HashMap<String, FieldMeta> getFieldMetaMap = new HashMap<>();

            for (int i = 0; i < colCount; i++) {
                FieldMeta fieldmeta = new FieldMeta();
                fieldmeta.setTable_name(rsmd.getTableName(i + 1));
                fieldmeta.setColumn_name(rsmd.getColumnName(i + 1));
                fieldmeta.setPrecision(rsmd.getPrecision(i + 1));
                fieldmeta.setScale(rsmd.getScale(i + 1));
                if (DBProdName.equals("MySQL")) {
                    Field[] fields = ((com.mysql.cj.jdbc.result.ResultSetMetaData) rsmd).getFields();
                    if (fields.length > i) {
                        fieldmeta.setTable_alias(fields[i].getTableName());
                    }
                }
                String Columnlabel = rsmd.getColumnLabel(i + 1);
                fieldmeta.setColumn_label(Columnlabel);
                colNameList.add(Columnlabel);


                switch (rsmd.getColumnTypeName(i + 1).toLowerCase()) {
                    case "int":
                    case "smallint": {
                        fieldmeta.setFieldtype(Integer.class);
                        break;
                    }
                    case "bigint": {
                        fieldmeta.setFieldtype(Long.class);
                        break;
                    }
                    case "float":
                    case "numeric":
                    case "decimal": {
                        fieldmeta.setFieldtype(BigDecimal.class);
                        break;
                    }
                    case "date":
                    case "datetime": {
                        fieldmeta.setFieldtype(Date.class);
                        break;
                    }
                    case "boolean": {
                        fieldmeta.setFieldtype(Boolean.class);
                        break;
                    }
                    case "json": {
                        fieldmeta.setFieldtype(JSON.class);
                        break;
                    }
                    default: {
                        fieldmeta.setFieldtype(String.class);
                        break;
                    }
                }
                getFieldMetaMap.put(Columnlabel, fieldmeta);
            }
            boolean addrowindex = !colNameList.contains("rowindex");
            int rowindex = 1;
            long pageNumber = -1;
            long pageSize = -1;
            while (resultSet.next()) {
                Row row = new Row();
                for (int i = 0; i < colCount; i++) {
                    String key = colNameList.get(i);
                    Class clazz = getFieldMetaMap.get(key).getFieldtype();
                    Object value = null;
                    if (clazz == Integer.class) {
                        value = resultSet.getInt(colNameList.get(i));
                    } else if (clazz == Long.class) {
                        value = resultSet.getLong(colNameList.get(i));
                    } else if (clazz == BigDecimal.class) {
                        value = resultSet.getBigDecimal(colNameList.get(i));
                    } else if (clazz == Boolean.class) {
                        value = resultSet.getBoolean(colNameList.get(i));
                    } else if (clazz == JSON.class) {
                        value = JSON.parse(resultSet.getString(colNameList.get(i)));
                    } else if (clazz == Date.class) {
                        value = resultSet.getString(colNameList.get(i));
                        if (value != null) {
                            value = ((String) value).replace(" 00:00:00.0", "");
                        }
                    } else {
                        Object objvalue = resultSet.getObject(colNameList.get(i));
                        value = (objvalue == null) ? "" : objvalue;
                    }
                    if ("total".equals(key)) {
                        value = value == null ? 0L : value;
                        total = total < 0 ? (Long) value : total;
                    } else if ("pageTotal".equals(key)) {
                        if (value instanceof Long) {
                            pageTotal = pageTotal < 0 ? (Long) value : pageTotal;
                        } else if (value instanceof BigDecimal) {
                            pageTotal = pageTotal < 0 ? ((BigDecimal) value).longValue() : pageTotal;
                        } else if (value != null) {
                            pageTotal = pageTotal < 0 ? Long.parseLong(value.toString()) : pageTotal;
                        }
                    } else if ("pageNumber".equals(key)) {
                        value = value == null ? 0L : value;
                        pageNumber = (Long) value;
                    } else if ("pageSize".equals(key)) {
                        value = value == null ? 0L : value;
                        pageSize = (Long) value;
                    } else {
                        row.put(key, value);
                    }
                }
                if (addrowindex) {
                    if (pageNumber > 0 && pageSize > 0) {
                        row.put("rowindex", ((pageNumber - 1) * pageSize) + (rowindex++));
                    } else {
                        row.put("rowindex", rowindex++);
                    }
                }
                row.setFieldList(colNameList);
                row.setFieldMetaMap(getFieldMetaMap);
                rows.add(row);
            }
            if (total > 0) {
                rows.totalRows = total;
            }
            if (pageTotal > 0) {
                rows.totalPage = pageTotal;
            } else if (pageSize > 0) {
                rows.totalPage = new Double(Math.ceil((double) total / (double) pageSize)).longValue();
            }
            if (rows.totalRows == 0) {
                rows.totalRows = rows.size();
                rows.totalPage = 1;
            }
            rows.setFieldList(colNameList);
            rows.setFieldMetaMap(getFieldMetaMap);
        } catch (SQLException e) {
            if (createsqllog) {
                sqlLogList.add(new SQLLog(SQL, databasekey + "查询语句执行错误：" + e.getMessage()));
            }
            throw new YosException(databasekey + "查询语句执行错误：" + SQL + System.lineSeparator() + e.getMessage());
        } finally {
            if (resultSet != null) {
                try {
                    resultSet.close();
                } catch (SQLException e) {
                    throw new YosException(e.getMessage());
                }
            }
            if (found_rowsResultSet != null) {
                try {
                    found_rowsResultSet.close();
                } catch (SQLException e) {
                    throw new YosException(e.getMessage());
                }
            }
            close(yosConnection, statement);
        }
        if (createsqllog) {
            sqlLogList.add(new SQLLog(SQL, Calendar.getInstance().getTimeInMillis() - starttimes, rows.size()));
        }
        return rows;
    }

    /**
     * 执行sql更新语句
     * @param sqlFactory
     * @return
     */
    public void runSqlUpdate(SQLFactory sqlFactory) throws YosException {
        runSqlUpdate(sqlFactory.getSQL());
    }

    /**
     * 执行sql更新语句
     * @param SQL
     * @return 受影响的行数
     */
    public int runSqlUpdate(String SQL) throws YosException {
        ArrayList<String> list = new ArrayList<>();
        list.add(SQL);
        return runSqlUpdate(list);
    }

    /**
     * 执行sql更新语句组
     * @param SQLlist
     * @return 受影响的数据行数
     */
    public int runSqlUpdate(ArrayList<String> SQLlist) throws YosException {
        int affectedrows = 0;
        boolean success = true;
        long starttimes = Calendar.getInstance().getTimeInMillis();
        DBConnectPool.YosConnection yosConnection = new DBConnectPool().getConnect(databasekey);
        Connection conn = yosConnection.getConnection();
        if (conn == null) {
            if (createsqllog) {
                for (String SQL : SQLlist) {
                    sqlLogList.add(new SQLLog(SQL, databasekey + "数据库连接失败"));
                }
            }
            throw new YosException(databasekey + "数据库连接失败");
        }
        PreparedStatement statement = null;
        try {
            for (String sql : SQLlist) {
                statement = conn.prepareStatement(sql);
                affectedrows = affectedrows + statement.executeUpdate();
            }
            conn.commit();
        } catch (SQLException e) {
            success = false;
            String errmsg = e.getMessage();
            try {
                conn.rollback();
            } catch (SQLException e1) {
                throw new YosException(e1.getMessage());
            }
            if (createsqllog) {
                if (SQLlist.size() == 1) {
                    sqlLogList.add(new SQLLog(SQLlist.get(0), databasekey + "更新语句执行错误：" + e.getMessage()));
                } else if (SQLlist.size() > 1) {
                    sqlLogList.add(new SQLLog(JSON.toJSON(SQLlist).toString(), databasekey + "更新语句执行错误：" + e.getMessage()));
                }
            }
            if (errmsg.startsWith("Duplicate entry") && errmsg.endsWith(".PRIMARY'")) {
                errmsg = errmsg.replace("Duplicate entry '", "").replace("' for key '", ",").replace(".PRIMARY'", "");
                String[] s = errmsg.split(",");
                throw new YosException("表" + s[1] + "主键 ID【" + s[0] + "】重复！");
            } else if (errmsg.startsWith("Duplicate entry")) {
                errmsg = errmsg.replace("Duplicate entry '", "").replace("' for key '", ",").replace("'", "");
                String[] s = errmsg.split(",");
                throw new YosException("数据【" + s[0] + "】触发了唯一索引" + s[1]);
            } else if (errmsg.startsWith("DataVersion conflict")) {
                throw new YosException("DataVersion conflict");
            } else {
                throw new YosException(databasekey + "更新语句执行错误：" + JSON.toJSON(SQLlist) + System.lineSeparator() + errmsg);
            }
        } finally {
            close(yosConnection, statement);
        }
        if (success && createsqllog) {
            if (SQLlist.size() == 1) {
                sqlLogList.add(new SQLLog(SQLlist.get(0), Calendar.getInstance().getTimeInMillis() - starttimes, affectedrows));
            } else if (SQLlist.size() > 1) {
                sqlLogList.add(new SQLLog(JSON.toJSON(SQLlist).toString(), Calendar.getInstance().getTimeInMillis() - starttimes, affectedrows));
            }
        }
        return affectedrows;
    }

    /**
     * 数据库连接释放
     * @param yosConnection
     * @param statement
     */
    private void close(DBConnectPool.YosConnection yosConnection, PreparedStatement statement) {
        if (statement != null) {
            try {
                statement.close();
                statement = null;
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        new DBConnectPool().close(yosConnection);
    }

    public String getDateTime_Str() {
        return BaseClass.dateTimeFormat.format(Calendar.getInstance().getTime());
    }

}
