package utility.valve;

import common.BaseClass;
import org.apache.commons.math3.fitting.PolynomialCurveFitter;
import org.apache.commons.math3.fitting.WeightedObservedPoint;

import java.util.ArrayList;
import java.util.Collection;

public class Valve extends BaseClass {
    private ValvePot valvePot;
    private long itemid = 0L;

    private double D;//口径，根据节流件id查询获得
    private double kvmax = 0;//节流件最大kv值，根据节流件id查询获得

    private double[] LiArray;//阀门开度值
    private double[] KvArray;//kv系数
    private double[] DcArray;//气蚀系数

    /**
     * @param itemid 节流件id
     */
    public Valve(long itemid, ValvePot valvePot) {
        this.itemid = itemid;
        this.D = 1200;//根据节流件id查询获得
        this.kvmax = 30960;//根据节流件id查询获得
        this.LiArray = new double[]{0.0, 0.05, 0.1, 0.15, 0.18, 0.2, 0.22, 0.25, 0.28, 0.3, 0.32, 0.35, 0.4, 0.45, 0.5, 0.55, 0.6, 0.65, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 1.0};
        this.KvArray = new double[]{0.0, 1265.0, 2668.0, 4063.0, 4894.0, 5450.0, 6000.0, 6810.0, 7630.0, 8160.0, 8960.0, 10150.0, 14650.0, 18720.0, 22310.0, 25420.0, 28090.0, 30350.0, 30960.0, 30960.0, 30960.0, 30960.0, 30960.0, 30960.0, 30960.0};
        this.DcArray = new double[]{0.0, 0.1082, 0.2151, 0.3022, 0.348, 0.3758, 0.3996, 0.4338, 0.4625, 0.4797, 0.5051, 0.7, 1.1035, 1.1553, 1.1977, 1.2289, 1.2537, 1.2714, 1.275, 1.275, 1.275, 1.275, 1.275, 1.275, 1.275};
        this.valvePot = valvePot;
    }

    /**
     * 根据节流件阀门开度获取kv值
     *
     * @param li 阀门开度
     * @return
     */
    public double getItemKV(double li) {
        double kv = -1;
        for (int i = 0; i < LiArray.length - 1; i++) {
            if (LiArray[i] == li) {
                kv = KvArray[i];
                break;
            } else if (LiArray[i + 1] == li) {
                kv = KvArray[i + 1];
                break;
            } else if ((li - LiArray[i]) * (li - LiArray[i + 1]) < 0) {
                //如果li值在两个点之间，则进行曲线拟合计算
                kv = getPointY(LiArray[i], KvArray[i], LiArray[i + 1], KvArray[i + 1], li);
                break;
            }
        }
        return kv;
    }

    /**
     * 根据节流件阀门开度获取气蚀系数值
     *
     * @param li
     * @return
     */
    public double getItemDc(double li) {
        double dc = -1;
        for (int i = 0; i < LiArray.length - 1; i++) {
            if (LiArray[i] == li) {
                dc = DcArray[i];
                break;
            } else if (LiArray[i + 1] == li) {
                dc = DcArray[i + 1];
                break;
            } else if ((li - LiArray[i]) * (li - LiArray[i + 1]) < 0) {
                //如果li值在两个点之间，则进行曲线拟合计算
                dc = getPointY(LiArray[i], DcArray[i], LiArray[i + 1], DcArray[i + 1], li);
                break;
            }
        }
        return dc;
    }

    /**
     * 根据kv值计算阀门开度
     *
     * @param kv
     * @return
     */
    public double calcLi(double kv) {
        double li = -1;
        for (int i = 0; i < LiArray.length - 1; i++) {
            if (KvArray[i] == kv) {
                li = LiArray[i];
                break;
            } else if (KvArray[i + 1] == kv) {
                li = LiArray[i + 1];
                break;
            } else if ((kv - KvArray[i]) * (kv - KvArray[i + 1]) < 0) {
                //如果kv值在两个点之间，则进行曲线拟合计算
                li = getPointX(LiArray[i], KvArray[i], LiArray[i + 1], KvArray[i + 1], kv);
                break;
            }
        }
        return li;
    }

    /**
     * 根据流量计算管道流速
     *
     * @param q 流量
     */
    public double calcV(double q) {
        q = ValveUnit.getQUnitValue(valvePot.getQ_unit(), q, ValveUnit.QUnit.m3s);//将流量转换为立方米每秒
        double v = (4.0 * 1000d * 1000d * q) / (Math.PI * Math.pow(D, 2));
        return ValveUnit.getVUnitValue(ValveUnit.VUnit.ms, v, valvePot.getV_unit());
    }


    /**
     * 根据阀前阀后压力差及kv值 计算管道流量
     *
     * @param p  阀前后压力差
     * @param Kv kv值
     * @return
     */
    public double calcQ(double p, double Kv) {
        p = ValveUnit.getPUnitValue(valvePot.getP_unit(), p, ValveUnit.PUnit.kPa);//将阀前压力转换为kPa
        double ρ = valvePot.getρ();//介质密度
        double q = Kv / (Math.sqrt(ρ / p) * 10d);
        return ValveUnit.getQUnitValue(ValveUnit.QUnit.m3h, q, valvePot.getQ_unit());
    }

    /**
     * 根据阀前阀后压力及流量 计算kv值
     *
     * @param p 阀前后压力差
     * @param q 过阀流量
     * @return
     */
    public int calcKV(double p, double q) {
        p = ValveUnit.getPUnitValue(valvePot.getP_unit(), p, ValveUnit.PUnit.kPa);//将阀前压力转换为kPa
        double ρ = valvePot.getρ();//介质密度
        q = ValveUnit.getQUnitValue(valvePot.getQ_unit(), q, ValveUnit.QUnit.m3h);//将流量转换为立方米每小时
        double kv = (10d * q) * Math.sqrt(ρ / p);
        return new Double(Math.ceil(kv)).intValue();
    }


    /**
     * 根据阀前阀后压力及流量计算汽蚀系数
     *
     * @param p1 阀前压力
     * @param p2 阀后压力
     * @param v  流速
     * @return
     */
    public double calcDc(double p1, double p2, double v) {
        p1 = ValveUnit.getPUnitValue(valvePot.getP_unit(), p1, ValveUnit.PUnit.mH2O);//将阀前压力转换为mH2O
        p2 = ValveUnit.getPUnitValue(valvePot.getP_unit(), p2, ValveUnit.PUnit.mH2O);//将阀后压力转换为mH2O
        double Pa = valvePot.getPa(ValveUnit.PUnit.mH2O);//获取大气压
        double Pv = valvePot.getPv(ValveUnit.PUnit.mH2O);//获取饱和蒸气压
        double g = valvePot.getG();//获取重力加速度
        double V2 = ValveUnit.getVUnitValue(valvePot.getV_unit(), v, ValveUnit.VUnit.ms);//获取管道流速，并将流速转换为米/秒
        double dc = (p2 + Pa - Pv) / ((p1 - p2) + (Math.pow(V2, 2) * 10) / (2.0 * g));
        System.err.println(p2 + Pa - Pv);
        return dc;
    }

    /**
     * 根据阀前压力及流量和气蚀系数计算阀后压力
     *
     * @param p1 阀前压力
     * @param v  流速
     * @param dc 临界气蚀系数
     * @return
     */
    public double calcP2(double p1, double v, double dc) {
        p1 = ValveUnit.getPUnitValue(valvePot.getP_unit(), p1, ValveUnit.PUnit.mH2O);//将阀前压力转换为mH2O
        double Pa = valvePot.getPa(ValveUnit.PUnit.mH2O);//获取大气压
        double Pv = valvePot.getPv(ValveUnit.PUnit.mH2O);//获取饱和蒸气压
        double g = valvePot.getG();//获取重力加速度
        double V2 = ValveUnit.getVUnitValue(valvePot.getV_unit(), v, ValveUnit.VUnit.ms);//获取管道流速，并将流速转换为米/秒
        double pc = (p1 + Pa - Pv - ((Math.pow(V2, 2) * dc * 10) / (2.0 * g))) / (1.0 + dc);
        return p1 - pc;
    }

    /**
     * 根据阀前压力及流量和气蚀系数计算阀前压力
     *
     * @param p2 阀后压力
     * @param q  最大流量
     * @param dc 临界气蚀系数
     * @return
     */
    public double calcP1(double p2, double q, double dc) {
        p2 = ValveUnit.getPUnitValue(valvePot.getP_unit(), p2, ValveUnit.PUnit.mH2O);//将阀前压力转换为mH2O
        double Pa = valvePot.getPa(ValveUnit.PUnit.mH2O);//获取大气压
        double Pv = valvePot.getPv(ValveUnit.PUnit.mH2O);//获取饱和蒸气压
        double g = valvePot.getG();//获取重力加速度
        double V2 = ValveUnit.getVUnitValue(valvePot.getV_unit(), calcV(q), ValveUnit.VUnit.ms);//获取管道流速，并将流速转换为米/秒
        double pc = ((p2 + Pa - Pv) / dc) - (Math.pow(V2, 2) * 10) / (2.0 * g);
        return p2 + pc;
    }


    public static void caculate(String name, double p1, double p2, double q) {
        ValvePot valvePot = new ValvePot(ValveUnit.PUnit.mH2O, ValveUnit.QUnit.m3h, ValveUnit.VUnit.ms);
        Valve valve = new Valve(1, valvePot);

        double kv = valve.calcKV(p1 - p2, q);//根据压差计算kv值
        double v = valve.calcV(q);//根据流量计算流速

        double dc = valve.calcDc(p1, p2, v);//计算气蚀系数
        double li = valve.calcLi(kv);//根据kv值计算开度
        double p = p1 - p2;//压差
        double l_dc = valve.getItemDc(li);//根据阀门开度计算临界气蚀系数

        double l_p2 = valve.calcP2(p1, v, l_dc);//根据临界气蚀系数，计算临界阀后压力
        double l_p = p1 - l_p2;//临界压差
        double fkv = Math.sqrt(p / l_p);//fkv

        System.err.println(name + ":");
        System.err.println("    阀前压力：" + p1 + " " + valvePot.getP_unit());
        System.err.println("    阀后压力：" + p2 + " " + valvePot.getP_unit());
        System.err.println("    压差：" + p + " " + valvePot.getP_unit());
        System.err.println("    流量：" + q + " " + valvePot.getQ_unit());
        System.err.println("    流速：" + v + " " + valvePot.getV_unit());
        System.err.println("    开度：" + li);
        System.err.println("    KV：" + kv);
        System.err.println("    气蚀系数：" + dc);
        System.err.println("    临界阀后压力：" + l_p2 + " " + valvePot.getP_unit());
        System.err.println("    临界压差：" + l_p + " " + valvePot.getP_unit());
        System.err.println("    临界气蚀系数：" + l_dc);
        System.err.println("    fkv：" + fkv);
    }

    public static void calulateByLi(String name, double p1, double p2, double li) {
        ValvePot valvePot = new ValvePot(ValveUnit.PUnit.mH2O, ValveUnit.QUnit.m3h, ValveUnit.VUnit.ms);
        Valve valve = new Valve(1, valvePot);
        double kv = valve.getItemKV(li);
        double q = valve.calcQ(p1 - p2, kv);
        caculate(name, p1, p2, q);
    }

    public static void main(String[] args) {
        caculate("小压差，小流量", 565, 535, 1108);
//        caculate("小压差，大流量", 565, 535, 1108);
//        caculate("大压差，小流量", 561, 541, 4000);
//        caculate("大压差，大流量", 561, 541, 8858);
//
//        calulateByLi("小压差，最大流量", 565, 535, 0.85);
//        calulateByLi("大压差，最大流量", 561, 541, 0.85);
//        calulateByLi("小压差，最小流量", 565, 535, 0.05);
//        calulateByLi("大压差，最小流量", 561, 541, 0.05);
//
//        calulateByLi("100%开度阀后压力", 561, 313.27, 1);
//        System.err.println("小压差，大流量:" + valve.calcDc(565, 535, 1108));
//        System.err.println("大压差，小流量:" + valve.calcDc(561, 541, 4000));
//        System.err.println("大压差，大流量:" + valve.calcDc(561, 541, 8858));
//        System.err.println("最大压差，最小过流量,指定的最小开度:" + valve.calcDc(561, 541, valve.calcQ(561 - 541, 1265)));
//        System.err.println("最小压差，最小过流量,指定的最小开度:" + valve.calcDc(565, 535, valve.calcQ(565 - 535, 1265)));
//        System.err.println("最大压差，最大过流量,指定的最大开度:" + valve.calcDc(561, 541, valve.calcQ(561 - 541, 30960)));
//        System.err.println("最小压差，最大过流量,指定的最大开度:" + valve.calcDc(565, 535, valve.calcQ(565 - 535, 30960)));
//
//        System.err.println("kv最大工况");
//        double q = valve.getQ(565, 535, valve.kvmax);//根据设备最大压差，最大kv值获取最大流量
//        valve.getD(565, 535, q);

//
//        System.err.println("kv最大工况");
//        double q2 = valve.getQ(Unit.PUnit.mH2O, 561, 541, valve.kvmax) * 3600;//根据设备最大kv值获取最大流量
//        System.err.println(q2);
//        valve.getD(Unit.PUnit.mH2O, 565, 535, Unit.QUnit.m3h, q2);


//        System.err.println("100%开度无气蚀最小出口压力");
//        valve.getD(Unit.PUnit.mH2O, 565, 321.66, Unit.QUnit.m3h, 53624.29);
//
//
//        System.err.println(valve.getQ(Unit.PUnit.mH2O, 561, 541, 30960));

    }


    /**
     * 根据两个参照坐标点及x点计算y点
     *
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @param pointx
     * @return
     */
    private double getPointY(double x1, double y1, double x2, double y2, double pointx) {
        // 定义数据点
        double[] x = {x1, x2};
        double[] y = {y1, y2};
        // 创建加权观测点集合
        Collection<WeightedObservedPoint> obs = new ArrayList<>();
        for (int n = 0; n < x.length; n++) {
            obs.add(new WeightedObservedPoint(1, x[n], y[n]));
        }
        // 创建多项式曲线拟合器对象
        PolynomialCurveFitter fitter = PolynomialCurveFitter.create(1);
        // 拟合曲线
        double[] coefficients = fitter.fit(obs);
        return coefficients[0] + coefficients[1] * pointx;
    }

    /**
     * 根据两个参照坐标点及y点计算x点
     *
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @param pointY
     * @return
     */
    private double getPointX(double x1, double y1, double x2, double y2, double pointY) {
        // 定义数据点
        double[] x = {x1, x2};
        double[] y = {y1, y2};
        // 创建加权观测点集合
        Collection<WeightedObservedPoint> obs = new ArrayList<>();
        for (int n = 0; n < x.length; n++) {
            obs.add(new WeightedObservedPoint(1, x[n], y[n]));
        }
        // 创建多项式曲线拟合器对象
        PolynomialCurveFitter fitter = PolynomialCurveFitter.create(1);
        // 拟合曲线
        double[] coefficients = fitter.fit(obs);
        return (pointY - coefficients[0]) / coefficients[1];
    }
}
