package common.data;

import com.alibaba.fastjson.JSONObject;
import common.Controller;
import common.YosException;
import common.data.db.DBConnect;

import java.util.*;

public class InsertSQL extends BaseSQL<InsertSQL> {

    protected InsertSQL(Controller controller, String tablename) throws YosException {
        super(controller, tablename);
    }

    protected InsertSQL(DBConnect dbConnect, String tablename) throws YosException {
        super(dbConnect, tablename);
    }

    /**
     * 根据row快速添加更新值
     *
     * @param row
     * @return
     * @throws YosException
     */
    public InsertSQL addRowValue(Row row) throws YosException {
        ArrayList<String> fieldlist = tableColumnNames.get(this.tablename);
        for (String fieldname : row.keySet()) {
            if (fieldlist.contains(fieldname)) {
                setValue(fieldname, row.get(fieldname));
            }
        }
        return this;
    }

    /**
     * 根据JSONObject快速添加更新值
     *
     * @param object
     * @return
     * @throws YosException
     */
    public InsertSQL addJSONValue(JSONObject object) throws YosException {
        ArrayList<String> fieldlist = tableColumnNames.get(this.tablename);
        for (String fieldname : object.keySet()) {
            if (fieldlist.contains(fieldname)) {
                setValue(fieldname, object.get(fieldname));
            }
        }
        return this;
    }

    @Override
    public InsertSQL setSiteid(String siteid) throws YosException {
        setValue("siteid", siteid);
        return this;
    }

    @Override
    public InsertSQL setUniqueid(long uniqueid) throws YosException {
        setValue(this.uniquecolumnname, uniqueid);
        return this;
    }

    /**
     * SQL参数Map
     */
    private HashMap<String, Object> keyvaluemap;

    /**
     * 为SQL实例添加文本类型参数值
     *
     * @param key
     * @param value
     */
    public InsertSQL setValue(String key, Object value) throws YosException {
        if (this.keyvaluemap == null) {
            this.keyvaluemap = new HashMap<>();
        }
        if (value instanceof Boolean) {
            value = (Boolean) value ? 1 : 0;
        }
        if (key.equalsIgnoreCase(this.uniquecolumnname) && (value == null || value.toString().equals("") || Long.parseLong(value.toString()) <= 0)) {
            value = createTableID(tablename);
        }
        this.keyvaluemap.put(key.toLowerCase(), value);
        return this;
    }

    /**
     * 为SQL实例添加参数，参数值会自动从controller.content中获取
     *
     * @param key
     * @return
     */
    public InsertSQL setValue(String key) throws YosException {
        if (this.controller != null && this.controller.content.containsKey(key)) {
            setValue(key, this.controller.content.get(key));
        }
        return this;
    }

    public InsertSQL setDateValue(String key) {
        if (this.keyvaluemap == null) {
            this.keyvaluemap = new HashMap<>();
        }
        this.keyvaluemap.put(key.toLowerCase(), Date.class);
        return this;
    }

    /**
     * 同表数据复制
     *
     * @param sourceid      源数据ID
     * @param excludeFields 需要排除的字段
     * @return
     */
    public InsertSQL copy(long sourceid, String... excludeFields) throws YosException {
        QuerySQL querySQL = new QuerySQL(dbConnect, tablename);
        querySQL.setUniqueid(sourceid);
        Rows rows = querySQL.query();
        if (rows.isNotEmpty()) {
            ArrayList<String> excludeFieldList = new ArrayList<>(Arrays.asList(excludeFields));
            excludeFieldList.add(uniquecolumnname);
            excludeFieldList.add("createby");
            excludeFieldList.add("createdate");
            excludeFieldList.add("createuserid");
            excludeFieldList.add("changeby");
            excludeFieldList.add("changedate");
            excludeFieldList.add("changeuserid");
            excludeFieldList.add("rowindex");
            excludeFieldList.add("createdepid");
            excludeFieldList.add("authdepid");
            excludeFieldList.add("authuserid");
            excludeFieldList.add(getdataversioncolumnname(tablename));
            Row row = rows.getRow(0);
            for (String fieldname : row.keySet()) {
                if (!excludeFieldList.contains(fieldname)) {
                    this.setValue(fieldname, row.get(fieldname));
                }
            }
        }
        return this;
    }

    /**
     * 异表数据复制
     *
     * @param sourceobject 数据源表
     * @param sourceid     源数据ID
     * @param copyfields   需要复制的字段，如两个表字段不一致，则可通过特殊符号进行标注如："objectname:tablename",冒号前为来源字段，后为当前表字段
     * @return
     */
    public InsertSQL copy(String sourceobject, long sourceid, String... copyfields) throws YosException {
        QuerySQL querySQL = new QuerySQL(dbConnect, sourceobject);
        querySQL.setUniqueid(sourceid);
        Rows sourceRows = querySQL.query();
        if (sourceRows.isNotEmpty()) {
            Row sourceRow = sourceRows.getRow(0);
            Set<String> rowketSet = sourceRow.keySet();
            for (String fieldname : copyfields) {
                if (fieldname.contains(":")) {
                    String[] fs = fieldname.split(":");
                    this.setValue(fs[1], sourceRow.get(fs[0]));
                } else if (rowketSet.contains(fieldname)) {
                    this.setValue(fieldname, sourceRow.get(fieldname));
                }
            }
        }
        return this;
    }

    /**
     * 获取SQL语句
     *
     * @return
     * @throws YosException
     */
    public String getSQL() throws YosException {
        addDefaultField();

        StringBuilder headsql = new StringBuilder("insert into ").append(tablename).append("(");
        StringBuilder valuesql = new StringBuilder("select ");
        boolean isfirstfield = true;
        if (!this.keyvaluemap.containsKey(this.uniquecolumnname)) {
            this.keyvaluemap.put(this.uniquecolumnname, createTableID(tablename));
        }
        this.keyvaluemap.remove(getdataversioncolumnname(tablename));

        for (String fieldname : this.keyvaluemap.keySet()) {
            Object value = this.keyvaluemap.get(fieldname);
            if (value == null || "null".equalsIgnoreCase(value.toString())) {
                headsql.append(isfirstfield ? "" : ",").append(fieldname);
                valuesql.append(isfirstfield ? "" : ",").append("NULL");
            } else {
                headsql.append(isfirstfield ? "" : ",").append(fieldname);
                if (value == Date.class) {
                    valuesql.append(isfirstfield ? "now()" : ",now()");
                } else {
                    if (value instanceof Date || value instanceof Calendar) {
                        Date date = value instanceof Date ? (Date) value : ((Calendar) value).getTime();

                        if ("date".equals(getColumnType(tablename, fieldname))) {
                            valuesql.append(isfirstfield ? "" : ",").append("'").append(getDate_Str(date)).append("'");
                        } else {
                            valuesql.append(isfirstfield ? "" : ",").append("'").append(getDateTime_Str(date)).append("'");
                        }
                    } else {
                        valuesql.append(isfirstfield ? "" : ",").append("'").append(value).append("'");
                    }
                }
            }
            isfirstfield = false;
        }
        headsql.append(")\r\n").append(valuesql);
        if (!wherelist.isEmpty()) {
            headsql.append("\r\nwhere ");
            boolean isfirstwhere = true;
            for (String where : wherelist) {
                headsql.append(isfirstwhere ? "(" : " and (").append(parameterDo(where)).append(")");
                isfirstwhere = false;
            }
        }
        return headsql.toString();
    }

    /**
     * 为插入语句设置默认字段，如果字段已被指定，则跳过。
     *
     * @throws YosException
     */
    private void addDefaultField() throws YosException {
        ArrayList<String> list = getTableColumnNames(tablename);
        if (controller != null) {
            if (list.contains("createuserid") && !keyvaluemap.containsKey("createuserid")) {
                this.setValue("createuserid", controller.userid);
            }
            if (list.contains("createby") && !keyvaluemap.containsKey("createby")) {
                this.setValue("createby", controller.username);
            }
            if (list.contains("createdepid") && !keyvaluemap.containsKey("createdepid")) {
                this.setValue("createdepid", controller.departmentid);
            }
            if (list.contains("changeuserid") && !keyvaluemap.containsKey("changeuserid")) {
                this.setValue("changeuserid", controller.userid);
            }
            if (list.contains("changeby") && !keyvaluemap.containsKey("changeby")) {
                this.setValue("changeby", controller.username);
            }
            if (list.contains("siteid") && !keyvaluemap.containsKey("siteid")) {
                this.setValue("siteid", controller.siteid);
            }
            if (list.contains("authdepid") && !keyvaluemap.containsKey("authdepid")) {
                this.setValue("authdepid", controller.departmentid);
            }
            if (list.contains("authuserid") && !keyvaluemap.containsKey("authuserid")) {
                this.setValue("authuserid", controller.userid);
            }
        }
        if (list.contains("changedate") && !keyvaluemap.containsKey("changedate")) {
            this.setDateValue("changedate");
        }
        if (list.contains("createdate") && !keyvaluemap.containsKey("createdate")) {
            this.setDateValue("createdate");
        }
    }


    private int effectrows = -1;//影响行数

    /**
     * 执行插入操作
     *
     * @return
     * @throws YosException
     */
    public int insert() throws YosException {
        if (effectrows == -1) {
            effectrows = dbConnect.runSqlUpdate(getSQL());
        }
        return effectrows;
    }

    public long getUniqueid() {
        return (Long) this.keyvaluemap.getOrDefault(this.uniquecolumnname, 0L);
    }

    private Row row = null;

    /**
     * 获取插入后的行，如果执行此方法前没有执行过插入操作，则会先执行插入操作
     *
     * @return
     * @throws YosException
     */
    public Row getRow() throws YosException {
        if (row == null) {
            if (effectrows == -1) {
                insert();
            }
            StringBuilder querySql = new StringBuilder("select * from ").append(tablename).append(" where ( 1=1 ) ");
            if (this.keyvaluemap.containsKey(uniquecolumnname)) {
                querySql.append(" and ").append(uniquecolumnname).append("=").append(this.keyvaluemap.get(uniquecolumnname));
                if (this.keyvaluemap.containsKey("siteid")) {
                    querySql.append(" and siteid='").append(this.keyvaluemap.get("siteid")).append("'");
                }
                Rows rows = dbConnect.runSqlQuery(querySql.toString());
                if (rows.size() == 1) {
                    row = rows.get(0);
                }
            }
        }
        return row == null ? new Row() : row;
    }
}
