package utility.tools;

import org.apache.commons.math3.fitting.PolynomialCurveFitter;
import org.apache.commons.math3.fitting.WeightedObservedPoint;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Random;

/**
 * 快速计算工具
 */
public class Math {

    /**
     * 加法
     *
     * @param arg0
     * @param arg1
     * @return
     */
    public static double add(double arg0, double arg1) {
        BigDecimal b1 = new BigDecimal(Double.toString(arg0));
        BigDecimal b2 = new BigDecimal(Double.toString(arg1));
        return b1.add(b2).doubleValue();
    }

    /**
     * 减法
     *
     * @param arg0 被减数
     * @param arg1 减数
     * @return
     */
    public static double sub(double arg0, double arg1) {
        BigDecimal b1 = new BigDecimal(Double.toString(arg0));
        BigDecimal b2 = new BigDecimal(Double.toString(arg1));
        return b1.subtract(b2).doubleValue();
    }

    /**
     * 乘法
     *
     * @param arg0
     * @param arg1
     * @return
     */
    public static double mul(double arg0, double arg1) {
        BigDecimal b1 = new BigDecimal(Double.toString(arg0));
        BigDecimal b2 = new BigDecimal(Double.toString(arg1));
        return b1.multiply(b2).doubleValue();
    }

    /**
     * 乘法
     *
     * @param arg0
     * @param arg1
     * @return
     */
    public static double mul(double arg0, double arg1, int scale) {
        BigDecimal b1 = new BigDecimal(Double.toString(arg0));
        BigDecimal b2 = new BigDecimal(Double.toString(arg1));

        return round(b1.multiply(b2).doubleValue(), scale);
    }

    /**
     * 除法，默认保留6位
     *
     * @param arg0
     * @param arg1
     * @return
     */
    public static double div(double arg0, double arg1) {
        return div(arg0, arg1, 6);
    }

    /**
     * 除法
     *
     * @param arg0  被除数
     * @param arg1  除数
     * @param scale 需要精确到小数点以后几位
     * @return
     */
    public static double div(double arg0, double arg1, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("精度不能小于0");
        }
        BigDecimal b1 = new BigDecimal(Double.toString(arg0));
        BigDecimal b2 = new BigDecimal(Double.toString(arg1));
        return b1.divide(b2, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 四舍五入处理。
     *
     * @param arg0  需要四舍五入的数字
     * @param scale 小数点后保留几位
     * @return 四舍五入后的结果
     */
    public static Double round(double arg0, int scale) {
        if (scale < 0) {
            throw new IllegalArgumentException("保留位数不能小于0");
        }
        BigDecimal b = new BigDecimal(Double.toString(arg0));
        BigDecimal one = new BigDecimal("1");
        return b.divide(one, scale, BigDecimal.ROUND_HALF_UP).doubleValue();
    }

    /**
     * 获取随机数
     *
     * @return
     */
    public static double random() {
        return java.lang.Math.random();
    }

    public static String random(int len) {
        Random r = new Random();
        StringBuilder rs = new StringBuilder();
        for (int i = 0; i < len; i++) {
            rs.append(r.nextInt(10));
        }
        return rs.toString();
    }

    /**
     * 计算百分比，保留2位数
     *
     * @param x
     * @param y
     * @return 50%
     */
    public static String getPercent(long x, long y) {
        double d1 = x * 1.0;
        double d2 = y * 1.0;
        NumberFormat percentInstance = NumberFormat.getPercentInstance();
        // 设置保留几位小数，这里设置的是保留两位小数
        percentInstance.setMinimumFractionDigits(2);
        return percentInstance.format(d1 / d2);
    }


    /**
     * 根据两个参照坐标点及x点计算y点
     *
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @param pointx
     * @return
     */
    public static double getPointY(double x1, double y1, double x2, double y2, double pointx) {
        // 定义数据点
        double[] x = {x1, x2};
        double[] y = {y1, y2};
        // 创建加权观测点集合
        Collection<WeightedObservedPoint> obs = new ArrayList<>();
        for (int n = 0; n < x.length; n++) {
            obs.add(new WeightedObservedPoint(1, x[n], y[n]));
        }
        // 创建多项式曲线拟合器对象
        PolynomialCurveFitter fitter = PolynomialCurveFitter.create(1);
        // 拟合曲线
        double[] coefficients = fitter.fit(obs);
        return coefficients[0] + coefficients[1] * pointx;
    }

    /**
     * 根据两个参照坐标点及y点计算x点
     *
     * @param x1
     * @param y1
     * @param x2
     * @param y2
     * @param pointY
     * @return
     */
    public static double getPointX(double x1, double y1, double x2, double y2, double pointY) {
        // 定义数据点
        double[] x = {x1, x2};
        double[] y = {y1, y2};
        // 创建加权观测点集合
        Collection<WeightedObservedPoint> obs = new ArrayList<>();
        for (int n = 0; n < x.length; n++) {
            obs.add(new WeightedObservedPoint(1, x[n], y[n]));
        }
        // 创建多项式曲线拟合器对象
        PolynomialCurveFitter fitter = PolynomialCurveFitter.create(1);
        // 拟合曲线
        double[] coefficients = fitter.fit(obs);
        return (pointY - coefficients[0]) / coefficients[1];
    }
}
