package common.data;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import common.YosException;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class Rows extends ArrayList<Row> {

    private static final long serialVersionUID = 1L;
    public long totalRows = 0;
    public long totalPage = 0;
    public long pageSize = 0;
    public long pageNumber = 0;
    public String listqueryid = "";

    private ArrayList<String> fieldlist;
    private HashMap<String, FieldMeta> fieldMetaMap;

    /**
     * 获取总行数，分页查询时有效
     *
     * @return
     */
    public long getTotalRows() {
        return totalRows;
    }

    /**
     * 获取总页数，，分页查询时有效
     *
     * @return
     */
    public long getTotalPage() {
        return totalPage;
    }

    /**
     * 获取指定的行
     *
     * @param index
     * @return
     */
    public Row getRow(int index) {
        return get(index);
    }


    /**
     * 快速获取指定字段值匹配到的第一行
     *
     * @param fieldname
     * @param value
     * @return
     */
    public Row getRow(String fieldname, Object value) {
        for (Row row : this) {
            if (row.getString(fieldname).equals(value.toString())) {
                return row;
            }
        }
        return null;
    }

    /**
     * 快速获取指定字段值匹配到的所有行
     *
     * @param fieldname
     * @param value
     * @return
     */
    public Rows getRows(String fieldname, Object value) {
        Rows rows = new Rows();
        rows.setFieldList(fieldlist);
        rows.setFieldMetaMap(fieldMetaMap);
        for (Row row : this) {
            if (row.getString(fieldname).equals(value.toString())) {
                rows.add(row);
            }
        }
        return rows;
    }

    /**
     * 获取指定的行
     *
     * @param index
     * @return
     */
    @Override
    public Row get(int index) {
        if (index + 1 > this.size()) {
            return null;
        }
        return super.get(index);
    }


    /**
     * 获取最后一行
     *
     * @return
     */
    public Row getLastRow() {
        return get(this.size() - 1);
    }

    /**
     * 移除指定的字段
     *
     * @param columns
     * @return
     */
    public Rows removeColumn(String... columns) {
        for (Row row : this) {
            for (String column : columns) {
                row.remove(column);
            }
        }
        return this;
    }

    /**
     * 移除指定的字段
     *
     * @param columns
     * @return
     */
    public Rows removeColumn(ArrayList<String> columns) {
        return removeColumn(columns.toArray(new String[]{}));
    }

    /**
     * 将当前rows转换成JsonArray
     *
     * @return
     */
    public JSONArray toJsonArray() {
        return (JSONArray) JSON.toJSON(this);
    }

    /**
     * 将当前rows的指定字段转换成JsonArray，默认去重
     *
     * @return ['A','B']
     */
    public JSONArray toJsonArray(String column) {
        return toJsonArray(column, true);
    }

    /**
     * 将当前rows的指定字段转换成JsonArray
     *
     * @param column     字段名称
     * @param isdistinct 是否去重
     * @return
     */
    public JSONArray toJsonArray(String column, boolean isdistinct) {
        ArrayList<Object> list = toArrayList(column, new ArrayList<>(), isdistinct);
        return (JSONArray) JSONArray.toJSON(list);
    }

    /**
     * 将当前rows的指定字段转换成数组，默认去重
     *
     * @param column 字段名称
     * @return
     */
    public String[] toArray(String column) {
        return toArrayList(column, true).toArray(new String[0]);
    }

    /**
     * 将当前rows的指定字段转换成数组
     *
     * @param column     字段名称
     * @param isdistinct 是否去重
     * @return
     */
    public String[] toArray(String column, boolean isdistinct) {
        return toArrayList(column, isdistinct).toArray(new String[0]);
    }

    /**
     * 将当前rows的指定字段转换成 ArrayList，默认去重
     *
     * @param column 字段名称
     * @return
     */
    public ArrayList<String> toArrayList(String column) {
        return toArrayList(column, true);
    }

    /**
     * 将当前rows的指定字段转换成 ArrayList，默认去重
     *
     * @param column     字段名称
     * @param isdistinct 是否去重
     * @return
     */
    public ArrayList<String> toArrayList(String column, boolean isdistinct) {
        ArrayList<String> list = new ArrayList<>();
        for (Row row : this) {
            list.add(row.getString(column));
        }
        return isdistinct ? new ArrayList<>(new LinkedHashSet<>(list)) : list;
    }

    /**
     * 将指定的字段值转换成数组
     *
     * @param column
     * @param array
     * @param <T>
     * @return
     */
    public <T> T[] toArray(String column, T[] array) {
        return toArray(column, array, true);
    }

    /**
     * 将指定的字段值转换成数组
     *
     * @param column
     * @param array
     * @param isdistinct
     * @param <T>
     * @return
     */
    public <T> T[] toArray(String column, T[] array, boolean isdistinct) {
        return toArrayList(column, new ArrayList<T>(), isdistinct).toArray(array);
    }

    /**
     * 将指定的字段值转换成ArrayList
     *
     * @param column
     * @param list
     * @param <T>
     * @return
     */
    public <T> ArrayList<T> toArrayList(String column, ArrayList<T> list) {
        return toArrayList(column, list, true);
    }

    /**
     * 将指定的字段值转换成ArrayList
     *
     * @param column
     * @param list
     * @param isdistinct
     * @param <T>
     * @return
     */
    public <T> ArrayList<T> toArrayList(String column, ArrayList<T> list, boolean isdistinct) {
        for (Row row : this) {
            Object value = row.get(column);
            list.add((T) value);
        }
        return isdistinct ? new ArrayList<>(new LinkedHashSet<>(list)) : list;
    }

    /**
     * 将Rows值转换成ArrayList<ArrayList<Object>>格式
     *
     * @return
     */
    public ArrayList<ArrayList<Object>> toListList() {
        ArrayList<ArrayList<Object>> list = new ArrayList<>();
        for (Row row : this) {
            list.add(row.toList(getFieldList()));
        }
        return list;
    }

    /**
     * 将Rows转换成RowsMap
     *
     * @param fieldnames
     * @return
     */
    public RowsMap toRowsMap(String... fieldnames) {
        RowsMap map = new RowsMap();
        for (Row row : this) {
            StringBuilder key = new StringBuilder();
            for (String fieldname : fieldnames) {
                key.append(row.getString(fieldname));
            }
            Rows subrows = map.getOrDefault(key.toString(), new Rows().setFieldMetaMap(getFieldMetaMap()).setFieldList(getFieldList()));
            subrows.add(row);
            map.put(key.toString(), subrows);
        }
        return map;
    }

    /**
     * 将Rows转换成xml数据格式
     *
     * @return
     */
    public String toXml() {
        Document document = DocumentHelper.createDocument();
        if (this.size() > 0) {
            Iterator<String> it = this.get(0).keySet().iterator();
            ArrayList<String> list = new ArrayList<String>();
            while (it.hasNext()) {
                list.add(it.next());
            }
            Element root = document.addElement("rows");
            for (int i = 0; i < this.size(); i++) {
                Row row = getRow(i);
                Element element = root.addElement("row" + i);
                for (String key : list) {
                    Object value = row.get(key);
                    Element e = element.addElement(key);
                    if (value == null) {
                        e.addText("null");
                    } else {
                        e.addText(row.get(key).toString());
                    }
                }
            }
        }
        return document.asXML();
    }

    /**
     * 判断是否存在数据
     *
     * @return
     */
    public boolean isNotEmpty() {
        return !this.isEmpty();
    }

    public Rows setFieldList(ArrayList<String> fieldlist) {
        this.fieldlist = fieldlist;
        return this;
    }

    /**
     * 获取Rows所有的字段
     *
     * @return
     */
    public ArrayList<String> getFieldList() {
        return this.fieldlist;
    }

    public HashMap<String, FieldMeta> getFieldMetaMap() {
        return fieldMetaMap;
    }

    public FieldMeta getFieldMeta(String fieldname) {
        return fieldMetaMap.get(fieldname);
    }

    /**
     * 获取当前Rows数据所在的表
     *
     * @return
     */
    public ArrayList<String> getTableNames() {
        ArrayList<String> arrayList = new ArrayList<>();
        for (FieldMeta fieldMeta : fieldMetaMap.values()) {
            if (!arrayList.contains(fieldMeta.getTable_name())) {
                arrayList.add(fieldMeta.getTable_name());
            }
        }
        return arrayList;
    }

    public Rows setFieldMetaMap(HashMap<String, FieldMeta> fieldMetaMap) {
        this.fieldMetaMap = fieldMetaMap;
        return this;
    }

    @Deprecated
    public void setFieldTypeMap(HashMap<String, Class> keytypemap) {
        if (this.fieldMetaMap == null) {
            this.fieldMetaMap = new HashMap<>();
        }
        for (String fieldname : keytypemap.keySet()) {
            FieldMeta fieldMeta = null;
            if (this.fieldMetaMap.containsKey(fieldname)) {
                fieldMeta = this.fieldMetaMap.get(fieldname);
            } else {
                fieldMeta = new FieldMeta();
            }
            fieldMeta.setFieldtype(keytypemap.get(fieldname));
            this.fieldMetaMap.put(fieldname, fieldMeta);
        }
    }

    /**
     * 数据排序
     *
     * @param fieldnames 排序字段 可设置多个排序字段。默认正序排序，如果需要倒序则指定排序方式如"fieldname desc"
     */
    public void sortby(String... fieldnames) {
        this.sort(new Comparator<Row>() {
            @Override
            public int compare(Row o1, Row o2) {
                int comparison = 0;
                for (String fieldname : fieldnames) {
                    if (comparison == 0) {
                        boolean asc = true;
                        if (fieldname.contains(" desc")) {
                            asc = false;
                        }
                        fieldname = fieldname.replace(" asc", "").replace(" desc", "").trim();
                        if (fieldMetaMap != null) {
                            Class fieldclazztype = fieldMetaMap.containsKey(fieldname) ? getFieldMeta(fieldname).getFieldtype() : String.class;
                            if (fieldclazztype == Integer.class) {
                                if (asc) {
                                    comparison = Integer.compare(o1.getInteger(fieldname), o2.getInteger(fieldname));
                                } else {
                                    comparison = Integer.compare(o2.getInteger(fieldname), o1.getInteger(fieldname));
                                }
                            } else if (fieldclazztype == Long.class) {
                                if (asc) {
                                    comparison = Long.compare(o1.getLong(fieldname), o2.getLong(fieldname));
                                } else {
                                    comparison = Long.compare(o2.getLong(fieldname), o1.getLong(fieldname));
                                }
                            } else if (fieldclazztype == Double.class) {
                                if (asc) {
                                    comparison = Double.compare(o1.getDouble(fieldname), o2.getDouble(fieldname));
                                } else {
                                    comparison = Double.compare(o2.getDouble(fieldname), o1.getDouble(fieldname));
                                }
                            } else if (fieldclazztype == Float.class) {
                                if (asc) {
                                    comparison = Float.compare(o1.getFloat(fieldname), o2.getFloat(fieldname));
                                } else {
                                    comparison = Float.compare(o2.getFloat(fieldname), o1.getFloat(fieldname));
                                }
                            } else if (fieldclazztype == Date.class) {
                                if (asc) {
                                    comparison = Long.compare(o1.getDate(fieldname).getTime(), o2.getDate(fieldname).getTime());
                                } else {
                                    comparison = Long.compare(o2.getDate(fieldname).getTime(), o1.getDate(fieldname).getTime());
                                }
                            } else {
                                if (asc) {
                                    comparison = o1.getString(fieldname).compareTo(o2.getString(fieldname));
                                } else {
                                    comparison = o2.getString(fieldname).compareTo(o1.getString(fieldname));
                                }
                            }
                        } else {
                            Object o1Object = o1.getOrDefault(fieldname, "");
                            if (o1Object instanceof Integer) {
                                if (asc) {
                                    comparison = Integer.compare(o1.getInteger(fieldname), o2.getInteger(fieldname));
                                } else {
                                    comparison = Integer.compare(o2.getInteger(fieldname), o1.getInteger(fieldname));
                                }
                            } else if (o1Object instanceof Long) {
                                if (asc) {
                                    comparison = Long.compare(o1.getLong(fieldname), o2.getLong(fieldname));
                                } else {
                                    comparison = Long.compare(o2.getLong(fieldname), o1.getLong(fieldname));
                                }
                            } else if (o1Object instanceof Double) {
                                if (asc) {
                                    comparison = Double.compare(o1.getDouble(fieldname), o2.getDouble(fieldname));
                                } else {
                                    comparison = Double.compare(o2.getDouble(fieldname), o1.getDouble(fieldname));
                                }
                            } else if (o1Object instanceof Float) {
                                if (asc) {
                                    comparison = Float.compare(o1.getFloat(fieldname), o2.getFloat(fieldname));
                                } else {
                                    comparison = Float.compare(o2.getFloat(fieldname), o1.getFloat(fieldname));
                                }
                            } else if (o1Object instanceof Date) {
                                if (asc) {
                                    comparison = Long.compare(o1.getDate(fieldname).getTime(), o2.getDate(fieldname).getTime());
                                } else {
                                    comparison = Long.compare(o2.getDate(fieldname).getTime(), o1.getDate(fieldname).getTime());
                                }
                            } else {
                                if (asc) {
                                    comparison = o1.getString(fieldname).compareTo(o2.getString(fieldname));
                                } else {
                                    comparison = o2.getString(fieldname).compareTo(o1.getString(fieldname));
                                }
                            }
                        }
                    }
                }
                return comparison;
            }
        });
    }

    /**
     * 为指定的字段进行语言翻译
     *
     * @param languagecode
     * @param fieldnames
     * @throws YosException
     */
    public void translate_simple(String languagecode, String... fieldnames) throws YosException {
        for (Row row : this) {
            row.translate_simple(languagecode, fieldnames);
        }
    }

    /**
     * 为指定的字段进行语言翻译
     *
     * @param languagecode
     * @param idcolumn
     * @param fieldnames
     * @throws YosException
     */
    public void translate(String languagecode, String idcolumn, String... fieldnames) throws YosException {
        for (Row row : this) {
            row.translate(languagecode, idcolumn, fieldnames);
        }
    }


    /**
     * 获取字段合计值
     *
     * @param column
     * @return
     */
    public BigDecimal sum(String column) {
        BigDecimal value = new BigDecimal("0");
        for (Row row : this) {
            if (row.containsKey(column)) {
                value = value.add(row.getBigDecimal(column));
            }
        }
        return value;
    }

    private HashMap<String, BigDecimal> minValueMap = new HashMap<String, BigDecimal>();
    private HashMap<String, BigDecimal> maxValueMap = new HashMap<String, BigDecimal>();

    /**
     * 获取字段最小值
     *
     * @param column
     * @return
     */
    public BigDecimal min(String column) {
        if (minValueMap.containsKey(column)) {
            return minValueMap.get(column);
        }
        caculate(column);
        return minValueMap.get(column);
    }

    /**
     * 获取字段最大值
     *
     * @param column
     * @return
     */
    public BigDecimal max(String column) {
        if (maxValueMap.containsKey(column)) {
            return maxValueMap.get(column);
        }
        caculate(column);
        return maxValueMap.get(column);
    }

    private void caculate(String column) {
        BigDecimal minvalue = new BigDecimal("0");
        BigDecimal maxvalue = new BigDecimal("0");
        for (Row row : this) {
            if (row.containsKey(column)) {
                BigDecimal value = row.getBigDecimal(column);
                minvalue = minvalue.min(value);
                maxvalue = maxvalue.max(value);
            }
        }
        minValueMap.put(column, minvalue);
        maxValueMap.put(column, maxvalue);
    }

    /**
     * 判断某个字段是否包含指定的值
     *
     * @param fieldname
     * @param value
     * @return
     */
    public boolean contains(String fieldname, Object value) {
        for (Row row : this) {
            if (row.getString(fieldname).equalsIgnoreCase(value.toString())) {
                return true;
            }
        }
        return false;
    }

    /**
     * 根据当前数据生成数据库插入语句
     *
     * @param tablename
     * @return
     * @throws YosException
     */
    public ArrayList<String> getInsertSQL(String tablename) throws YosException {
        ArrayList<String> list = new ArrayList<>();
        for (Row row : this) {
            list.add(row.getInsertSQL(tablename).getSQL());
        }
        return list;
    }

    /**
     * 根据当前数据生成数据库更新语句
     *
     * @param tablename
     * @return
     * @throws YosException
     */
    public ArrayList<String> getUpdateSQL(String tablename) throws YosException {
        ArrayList<String> list = new ArrayList<>();
        for (Row row : this) {
            list.add(row.getInsertSQL(tablename).getSQL());
        }
        return list;
    }

    /**
     * 合并rows中的row数据，将两个rows中的数据根据条件字段进行合并，当sql语句关联表多查询效率低时可使用此方法
     *
     * @param rows
     * @param keyfields
     * @return
     */
    public Rows merge(Rows rows, String... keyfields) {
        //如果为空，则直接返回
        if (rows.isEmpty()) {
            return this;
        }
        //清理重复和无用字段
        ArrayList<String> rc = new ArrayList<>();
        rc.add("rowindex");
        rc.add("pageNumber");
        rc.add("pageSize");
        if (this.fieldlist != null) {
            rc.addAll(this.fieldlist);
        }
        rc.addAll(this.get(0).keySet());
        rc.removeAll(Arrays.asList(keyfields));
        rows.removeColumn(rc);

        RowsMap rowsMap = rows.toRowsMap(keyfields);
        for (Row row : this) {
            StringBuilder key = new StringBuilder();
            for (String fieldname : keyfields) {
                key.append(row.getString(fieldname));
            }
            Rows keyrows = rowsMap.get(key);
            if (keyrows.isNotEmpty()) {
                row.putAll(keyrows.get(0));
            }
        }
        return this;
    }

    /**
     * 数据过滤,返回一个新的Rows对象
     *
     * @param key
     * @param value
     */
    public Rows filter(String key, Object value) {
        List<Row> rowlist = this.stream().filter(new Predicate<Row>() {
            @Override
            public boolean test(Row row) {
                return row.get(key).toString().equals(value.toString());
            }
        }).collect(Collectors.toList());
        Rows newRows = new Rows();
        newRows.addAll(rowlist);
        newRows.totalRows = newRows.size();
        newRows.totalPage = 1;
        newRows.fieldlist = this.fieldlist;
        newRows.fieldMetaMap = this.fieldMetaMap;
        return newRows;
    }
}
