/**
 *
 */
package common.data.db;

import common.BaseClass;
import common.YosException;
import common.data.Rows;
import common.parameter.properties;
import utility.tools.Math;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Properties;

/**
 * @author SJW
 *
 */
public class DBConnectPool {
    public static HashMap<String, Properties> propertiesMap = new HashMap<String, Properties>();
    /**
     * 数据库连接池
     */
    public static final HashMap<String, LinkedList<YosConnection>> POOLS_MAP = new HashMap<String, LinkedList<YosConnection>>();

    public static HashMap<String, LinkedList<String>> useingConnectionkeyMap = new HashMap<>();

    public YosConnection getConnect(String datakey) {
        synchronized (DBConnectPool.class) {
            YosConnection yosconnection = null;
            if (POOLS_MAP.containsKey(datakey) && (POOLS_MAP.get(datakey).size()) > 0) {
                yosconnection = POOLS_MAP.get(datakey).remove(0);
            } else if (!POOLS_MAP.containsKey(datakey)) {
                POOLS_MAP.put(datakey, new LinkedList<>());
                useingConnectionkeyMap.put(datakey, new LinkedList<>());
            }
            try {
                if (yosconnection == null || yosconnection.getConnection() == null || yosconnection.getConnection().isClosed()) {
                    yosconnection = buildConnect(datakey);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
            useingConnectionkeyMap.get(datakey).add(yosconnection.getConkey());
            return yosconnection;
        }
    }


    private YosConnection buildConnect(String datakey) throws YosException {
        synchronized (DBConnectPool.class) {
            YosConnection yosConnection;
            if (!propertiesMap.containsKey(datakey)) {
                Properties prop = getDataSource(datakey);
                if (prop != null && !prop.isEmpty()) {
                    propertiesMap.put(datakey, prop);
                }
            }
            Properties p = propertiesMap.get(datakey);
            try {
                Class.forName(p.getProperty("system.db.driver"));
                Connection conn = DriverManager.getConnection(p.getProperty("system.db.url"), p.getProperty("system.db.username"),
                        p.getProperty("system.db.password"));
                conn.setAutoCommit(false);
                conn.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
                yosConnection = new YosConnection(datakey, conn);
            } catch (Exception e) {
                e.printStackTrace();
                throw new YosException("创建" + datakey + "数据库连接失败！");
            }
            return yosConnection;
        }
    }

    public void close(YosConnection yosConnection) {
        synchronized (DBConnectPool.class) {
            try {
                if (yosConnection.isneedclose()) {
                    yosConnection.close();
                } else if (!yosConnection.isClosed()) {
                    POOLS_MAP.get(yosConnection.datakey).add(yosConnection);
                }
            } catch (SQLException e) {
                e.printStackTrace();
                System.err.println(yosConnection.datakey + "连接关闭失败" + getDateTime_Str() + e.getMessage());
            } finally {
                useingConnectionkeyMap.get(yosConnection.datakey).remove(yosConnection.getConkey());
            }
        }
    }

    public class YosConnection {
        private Connection connection;
        private Long createtime;
        private String datakey;
        private String conkey;

        private YosConnection(String datakey, Connection connection) {
            this.createtime = Calendar.getInstance().getTimeInMillis();
            this.datakey = datakey;
            this.connection = connection;
            this.conkey = this.datakey + "-" + createtime + "-" + Math.random(5);
        }

        public Connection getConnection() {
            return connection;
        }

        public String getDatakey() {
            return datakey;
        }

        public String getConkey() {
            return conkey;
        }

        public boolean isneedclose() {
            if ((Calendar.getInstance().getTimeInMillis() - this.createtime) > (3600 * 1000) || (POOLS_MAP.get(this.datakey).size() + useingConnectionkeyMap.get(this.datakey).size()) > 50) {
                return true;
            }
            try {
                if (!getConnection().isValid(1000)) {
                    return true;
                }
            } catch (Exception e) {
                return true;
            }
            return false;
        }

        private void close() throws SQLException {
            if (!this.isClosed()) {
                this.connection.close();
            }
        }

        public boolean isClosed() throws SQLException {
            return this.connection.isClosed();
        }
    }

    /**
     * 获取数据源
     * @param datakey
     * @return
     */
    private Properties getDataSource(String datakey) throws YosException {
        Properties prop = new Properties();
        switch (datakey) {
            case "default": {
                return properties.getYosProperties();
            }
            default: {
                DBConnect dbConnect = new DBConnect();
                String sql = "select driver,username,password,url from sys_datasource where datakey='" + datakey + "'";
                Rows rows = dbConnect.runSqlQuery(sql);
                if (!rows.isEmpty()) {
                    prop.setProperty("system.db.driver", rows.get(0).getString("driver"));
                    prop.setProperty("system.db.username", rows.get(0).getString("username"));
                    prop.setProperty("system.db.password", rows.get(0).getString("password"));
                    prop.setProperty("system.db.url", rows.get(0).getString("url"));
                } else {
                    return null;
                }
                break;
            }
        }
        return prop;
    }

    public String getDateTime_Str() {
        return BaseClass.dateTimeFormat.format(Calendar.getInstance().getTime());
    }
}
