package common.data;

import beans.parameter.Parameter;
import common.YosException;
import common.data.db.DBConnect;
import common.parameter.properties;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.*;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;

public class ExcelFactory {
    private String filename;
    private XSSFWorkbook xssfWorkbook;
    public Boolean isWrite = false;
    public static String filetype = "xlsx";

    /**
     * 构造函数，创建一个exlce输出文件
     *
     * @param filename
     */
    public ExcelFactory(String filename) {
        this.filename = filename + Calendar.getInstance().getTimeInMillis();
        xssfWorkbook = new XSSFWorkbook();
    }

    /**
     * 根据InputStream生成一个excel文件
     */
    public ExcelFactory(InputStream inputStream) throws IOException {
        this.filename = String.valueOf(Calendar.getInstance().getTimeInMillis());
        xssfWorkbook = new XSSFWorkbook(inputStream);
    }

    /**
     * 根据InputStream生成一个excel文件
     */
    public ExcelFactory(String filename, InputStream inputStream) throws IOException {
        this.filename = filename;
        xssfWorkbook = new XSSFWorkbook(inputStream);
    }

    /**
     * 创建一个空白的sheet
     *
     * @param sheetname sheet名称
     * @return
     */
    public XSSFSheet addSheet(String sheetname) {
        return xssfWorkbook.createSheet(sheetname);
    }

    /**
     * 创建一个带数据的sheet
     *
     * @param sheetname sheet名称
     * @param datarows  数据
     * @return
     */
    public XSSFSheet addSheet(String sheetname, Rows datarows) {
        return addSheet(sheetname, datarows, new HashMap<>(), new Integer[]{});
    }

    public XSSFSheet addSheet(String sheetname, Rows datarows, Integer[] columnwidths) {
        return addSheet(sheetname, datarows, new HashMap<>(), columnwidths);
    }

    public XSSFSheet addSheet(String sheetname, Rows datarows, HashMap<String, String> titlemap) {
        return addSheet(sheetname, datarows, titlemap, new Integer[]{});
    }

    /**
     * 创建一个带数据的sheet，并且可以指定标题名称
     *
     * @param sheetname sheet名称
     * @param datarows  数据
     * @param titlemap  标题重命名
     * @return
     */
    public XSSFSheet addSheet(String sheetname, Rows datarows, HashMap<String, String> titlemap, Integer[] columnwidths) {
        ArrayList<String> keylist = datarows.getFieldList();
        XSSFSheet sheet = xssfWorkbook.createSheet(sheetname);
        for (int i = 0; i < columnwidths.length && i < keylist.size(); i++) {
            int columnwidth = columnwidths[i];
            if (columnwidth == 0) {
                sheet.autoSizeColumn(i, true);
            } else {
                sheet.setColumnWidth(i, columnwidth * 1700);
            }
        }
        XSSFRow headrow = sheet.createRow(0);
        XSSFCellStyle xssfCellStyle = getHeadRowStyle();
        for (int i = 0; i < keylist.size(); i++) {
            XSSFCell xssFcell = headrow.createCell(i);
            xssFcell.setCellStyle(xssfCellStyle);
            String keyname = keylist.get(i);
            if (titlemap.containsKey(keyname)) {
                keyname = titlemap.get(keyname);
            }
            xssFcell.setCellValue(keyname);
        }
        for (int n = 0; n < datarows.size(); n++) {
            Row row = datarows.get(n);
            XSSFRow datarow = sheet.createRow(n + 1);
            for (int i = 0; i < keylist.size(); i++) {
                Class fieldclazztype = datarows.getFieldMeta(keylist.get(i)).getFieldtype();
                if (fieldclazztype == Integer.class) {
                    datarow.createCell(i).setCellValue(row.getInteger(keylist.get(i)));
                } else if (fieldclazztype == Long.class) {
                    datarow.createCell(i).setCellValue(row.getLong(keylist.get(i)));
                } else if (fieldclazztype == Float.class) {
                    datarow.createCell(i).setCellValue(row.getBigDecimal(keylist.get(i)).floatValue());
                } else if (fieldclazztype == Double.class) {
                    datarow.createCell(i).setCellValue(row.getBigDecimal(keylist.get(i)).doubleValue());
                } else {
                    datarow.createCell(i).setCellValue(row.getString(keylist.get(i)));
                }
            }
        }
        return sheet;
    }

    /**
     * 开始写入磁盘
     *
     * @throws IOException
     */
    public void write() throws IOException, YosException {
        FileOutputStream fileOutputStream = new FileOutputStream(new File(Parameter.get("system_excelexport_path") + this.filename + "." + filetype));
        xssfWorkbook.write(fileOutputStream);
        fileOutputStream.close();
        isWrite = true;
    }

    public String getFilename() {
        return this.filename + "." + filetype;
    }

    public String getFilepath() throws YosException {
        return Parameter.get("system_excelexport_path") + this.filename + "." + filetype;
    }

    public File getFile() throws YosException {
        return new File(Parameter.get("system_excelexport_path") + this.filename + "." + filetype);
    }

    public void deletefile() throws YosException {
        File file = new File(Parameter.get("system_excelexport_path") + this.filename + "." + filetype);
        file.delete();
    }

    /**
     * 默认标题样式
     *
     * @return
     */
    public XSSFCellStyle getHeadRowStyle() {
        XSSFCellStyle xssfCellStyle = xssfWorkbook.createCellStyle();
        xssfCellStyle.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        xssfCellStyle.setFillForegroundColor(IndexedColors.LIGHT_CORNFLOWER_BLUE.getIndex());//背景色
        xssfCellStyle.setAlignment(HorizontalAlignment.CENTER);//水平居中
        xssfCellStyle.setVerticalAlignment(VerticalAlignment.CENTER);//垂直居中
        XSSFFont font = xssfWorkbook.createFont();
        font.setFontHeightInPoints((short) 12);//字号
        font.setBold(true);//加粗
        xssfCellStyle.setFont(font);
        return xssfCellStyle;
    }

    public XSSFWorkbook getXssfWorkbook() {
        return this.xssfWorkbook;
    }

    /**
     * 获取sheet
     *
     * @param sheetname sheet名称
     * @return
     */
    public XSSFSheet getSheet(String sheetname) {
        return this.xssfWorkbook.getSheet(sheetname);
    }

    /**
     * 获取sheet
     *
     * @param sheetindex sheet位置
     * @return
     */
    public XSSFSheet getSheet(int sheetindex) {
        return this.xssfWorkbook.getSheetAt(sheetindex);
    }

    /**
     * 将sheet转换成Rows
     *
     * @param sheetname sheet名称
     * @param rows      从数据库查询结果为空的rows
     * @param index     开始行数
     * @return
     */
    public Rows getSheetRows(String sheetname, Rows rows, int index) {
        return getSheetRows(sheetname, rows.getFieldList(), index);
    }

    /**
     * 将sheet转换成Rows
     *
     * @param sheetindex sheet位置
     * @param rows       从数据库查询结果为空的rows
     * @param index      开始行数
     * @return
     */
    public Rows getSheetRows(int sheetindex, Rows rows, int index) {
        return getSheetRows(sheetindex, rows.getFieldList(), index);
    }


    /**
     * 将sheet转换成Rows
     *
     * @param sheetname sheet名称
     * @param fieldlist 字段key
     * @param rowindex  开始行数
     * @return
     */
    public Rows getSheetRows(String sheetname, ArrayList<String> fieldlist, int rowindex) {
        int sheetindex = this.xssfWorkbook.getSheetIndex(sheetname);
        return getSheetRows(sheetindex, fieldlist, rowindex);
    }

    /**
     * 将sheet转换成Rows
     *
     * @param sheetindex sheet位置
     * @param fieldlist  字段key
     * @param rowindex   开始行数
     * @return
     */
    public Rows getSheetRows(int sheetindex, ArrayList<String> fieldlist, int rowindex) {
        XSSFSheet sheet = this.xssfWorkbook.getSheetAt(sheetindex);
        int rowcount = sheet.getPhysicalNumberOfRows();
        Rows rows = new Rows();
        rows.setFieldList(fieldlist);
        for (; rowindex < rowcount; rowindex++) {
            Row row = new Row();
            XSSFRow xssfRow = sheet.getRow(rowindex);
            int columncount = xssfRow.getLastCellNum();
            for (int n = 0; n < columncount; n++) {
                XSSFCell cell = xssfRow.getCell(n);
                if (fieldlist.size() > n) {
                    String fieldname = fieldlist.get(n);
                    if (cell == null) {
                        row.put(fieldname, "");
                    } else {
                        try {
                            String celltype = cell.getCellType().name();
                            switch (celltype) {
                                case "STRING": {
                                    row.put(fieldname, cell.getStringCellValue());
                                    break;
                                }
                                case "NUMERIC": {
                                    row.put(fieldname, cell.getNumericCellValue());
                                    break;
                                }
                                default: {
                                    cell.setCellType(CellType.STRING);
                                    row.put(fieldname, cell.getStringCellValue());
                                    break;
                                }
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                            row.put(fieldname, "");
                        }
                    }
                }
            }
            rows.add(row);
        }
        return rows;
    }

    /**
     * 关闭
     */
    public void close() {
        if (xssfWorkbook != null) {
            try {
                xssfWorkbook.close();
                xssfWorkbook = null;
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public static void main(String[] args) throws YosException, IOException {
        DBConnect dbConnect = new DBConnect();
        Rows rows = dbConnect.runSqlQuery("select *from sys_users");
        ExcelFactory excelFactory = new ExcelFactory("asdas");
        HashMap<String, String> map = new HashMap<>();
        map.put("userid", "ID");
        map.put("name", "姓名");
        map.put("password", "密码");
        map.put("createby", "创建人");
        map.put("createdate", "创建时间");
        map.put("sysadministrator", "是否管理员");
        excelFactory.addSheet("nihao", rows, map, new Integer[]{1, 3, 5, 3, 3, 2, 3, 2, 2, 2, 3, 2, 2});
        excelFactory.write();
        excelFactory.close();
    }
}
