package common.data;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.parser.Feature;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

public class Row extends HashMap<String, Object> {

    private static final long serialVersionUID = 1L;

    /**
     * 转换为JsonObject
     *
     * @return
     */
    public JSONObject toJsonObject() {
        return (JSONObject) JSON.toJSON(this);
    }

    public Object getValueAsObject(String fieldname) {
        return get(fieldname);
    }

    public long getLong(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Long.parseLong(object.toString());
            }
            if (object instanceof Integer) {
                return Long.valueOf((Integer) object);
            }
            return (Long) object;
        } else {
            return 0L;
        }
    }

    public float getFloat(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Float.parseFloat(object.toString());
            }
            if (object instanceof Integer) {
                return (Float) object;
            }
            if (object instanceof BigDecimal) {
                return ((BigDecimal) object).floatValue();
            }
            if (object instanceof Float) {
                return (Float) object;
            }
            return (Float) object;
        } else {
            return 0F;
        }
    }

    public String getString(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object == null || String.valueOf(object).equalsIgnoreCase("null")) {
            return "";
        } else {
            if (object instanceof Float || object instanceof Double) {
                DecimalFormat df = new DecimalFormat("#");
                return df.format(object);
            }
            return String.valueOf(object);
        }
    }

    public JSONArray getJSONArray(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object instanceof String && object.toString().equals("")) {
            return new JSONArray();
        }
        try {
            return JSONArray.parseArray(object.toString());
        } catch (Exception e) {
            return new JSONArray();
        }
    }

    public Rows getRows(String linkname) {
        Object object = getValueAsObject(linkname);
        if (object instanceof Rows) {
            return (Rows) object;
        } else {
            return new Rows();
        }
    }

    public Row getRow(String linkname) {
        Object object = getValueAsObject(linkname);
        if (object instanceof Row) {
            return (Row) object;
        } else {
            return new Row();
        }
    }

    public JSONObject getJSONObject(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object instanceof String && object.toString().equals("")) {
            return new JSONObject();
        }
        try {
            return JSONObject.parseObject(object.toString(), Feature.OrderedField);
        } catch (Exception e) {
            return new JSONObject();
        }
    }

    public Integer getInteger(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Integer.parseInt(object.toString());
            }
            return (Integer) object;
        } else {
            return 0;
        }
    }

    public Short getShort(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return Short.parseShort(object.toString());
            }
            return (Short) object;
        } else {
            return 0;
        }
    }

    public double getDouble(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof String || object instanceof Integer) {
                return Double.parseDouble(object.toString());
            }
            if (object instanceof Double) {
                return (double) object;
            }
            if (object instanceof BigDecimal) {
                return ((BigDecimal) object).doubleValue();
            }
        }
        return 0d;
    }

    public BigDecimal getBigDecimal(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof String) {
                return new BigDecimal(object.toString());
            }
            if (object instanceof Integer) {
                return new BigDecimal((int) object);
            }
            if (object instanceof BigDecimal) {
                return (BigDecimal) object;
            }
        }
        return new BigDecimal("0");
    }

    public Boolean getBoolean(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof Boolean) {
                return (Boolean) object;
            }
            if (object instanceof String) {
                return "true".equalsIgnoreCase(object.toString()) || "1".equals(object.toString());
            }
            if (object instanceof Short) {
                return (Short) object == 1;
            }
            if (object instanceof Integer) {
                return (Integer) object == 1;
            }
            if (object instanceof Long) {
                return (Long) object == 1;
            }
        }
        return false;
    }

    public Date getDate(String fieldname) {
        Object object = getValueAsObject(fieldname);
        if (object != null) {
            if (object instanceof String) {
                String value = object.toString();
                if (value.length() == 10) {
                    value = value + " 00:00:00.0";
                }
                try {
                    return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse(value);
                } catch (ParseException e) {
                    e.printStackTrace();
                }
            }
            if (object instanceof Date) {
                return (Date) getValueAsObject(fieldname);
            }
            if (object instanceof Timestamp) {
                return new Date(((Timestamp) object).getTime());
            }
        }
        return null;
    }


    public String toXml() {
        Document document = DocumentHelper.createDocument();
        if (this.size() > 0) {
            Iterator<String> it = this.keySet().iterator();
            ArrayList<String> list = new ArrayList<String>();
            while (it.hasNext()) {
                list.add(it.next());
            }
            Element root = document.addElement("rows");
            Element element = root.addElement("row0");
            for (String key : list) {
                Object value = get(key);
                Element e = element.addElement(key);
                if (value == null) {
                    e.addText("null");
                } else {
                    e.addText(get(key).toString());
                }
            }
        }
        return document.asXML();
    }

    public <T> T toPOJO(T pojo) {
        try {
            Field[] pojofields = pojo.getClass().getFields();
            for (Field field : pojofields) {
                String fieldname = field.getName();
                if (this.containsKey(fieldname)) {
                    String fieldtype = field.getType().getSimpleName();
                    switch (fieldtype) {
                        case "String": {
                            field.set(pojo, this.getString(fieldname));
                            break;
                        }
                        case "Date": {
                            field.set(pojo, this.getDate(fieldname));
                            break;
                        }
                        case "Object": {
                            field.set(pojo, this.getValueAsObject(fieldname));
                            break;
                        }
                        case "int":
                        case "Integer": {
                            field.set(pojo, this.getInteger(fieldname));
                            break;
                        }
                        case "long":
                        case "Long": {
                            field.set(pojo, this.getLong(fieldname));
                            break;
                        }
                        case "short":
                        case "Short": {
                            field.set(pojo, this.getShort(fieldname));
                            break;
                        }
                        case "double":
                        case "Double": {
                            field.set(pojo, this.getBigDecimal(fieldname));
                            break;
                        }
                        case "boolean":
                        case "Boolean": {
                            field.set(pojo, this.getBoolean(fieldname));
                            break;
                        }
                        default:
                            break;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return pojo;
    }
}
