package common.data;

import com.influxdb.query.FluxRecord;
import com.influxdb.query.FluxTable;
import common.YosException;
import common.data.db.InfluxDBConnect;

import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

public class InfluxDBQuery {
    private final String siteid;
    private final long w_productid;
    private final long w_deviceid;
    private final String param;
    private String range_str = "";
    private String query_str = "";

    /**
     * @param siteid      站点 即bucket
     * @param w_productid 产品id 即_measurement
     * @param w_deviceid  设备id
     * @param param       参数
     */
    public InfluxDBQuery(String siteid, long w_productid, Long w_deviceid, String param) {
        this.siteid = siteid;
        this.w_productid = w_productid;
        this.w_deviceid = w_deviceid;
        this.param = param;
    }

    /**
     * 设置查询的时间范围
     *
     * @param timetype 时间类型
     * @param start    数据的开始时间，该值必须为负数，当timetype为d，start为-1时，即表示查询最近一天的意思
     * @return
     */
    public InfluxDBQuery setRange(TimeType timetype, int start) {
        this.range_str = "  |> range(start: " + start + timetype.toString() + ")\n";
        return this;
    }

    /**
     * 设置查询的时间范围,指定的开始时间到现在
     *
     * @param start 数据的开始时间
     * @return
     */
    public InfluxDBQuery setRange(Date start) {
        Calendar calendar_start = Calendar.getInstance();
        calendar_start.setTime(start);
        calendar_start.add(Calendar.HOUR, -8);

        this.range_str = "  |> range(start: " + new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'").format(calendar_start.getTime()) + ")\n";
        return this;
    }

    /**
     * 设置查询的时间范围
     *
     * @param timetype 时间类型
     * @param start    数据的开始时间，该值必须为负数，当timetype为d，start为-2，stop为-1时，即表示查询2天内且1天前的数据
     * @param stop     数据的结束时间，该值必须为负数，当timetype为d，start为-2，stop为-1时，即表示查询2天内且1天前的数据
     * @return
     */
    public InfluxDBQuery setRange(TimeType timetype, int start, int stop) {
        this.range_str = "  |> range(start: " + start + timetype.name() + ",stop:" + stop + timetype.name() + " )\n";
        return this;
    }

    /**
     * 设置查询的时间范围
     *
     * @param start 数据的开始时间
     * @param stop  数据的结束时间
     * @return
     */
    public InfluxDBQuery setRange(Date start, Date stop) {
        if (stop == null) {
            return setRange(start);
        }
        Calendar calendar_start = Calendar.getInstance();
        calendar_start.setTime(start);
        calendar_start.add(Calendar.HOUR, -8);

        Calendar calendar_end = Calendar.getInstance();
        calendar_end.setTime(stop);
        calendar_end.add(Calendar.HOUR, -8);

        this.range_str = "  |> range(start: " + new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'").format(calendar_start.getTime()) + ",stop:" + new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'").format(calendar_end.getTime()) + " )\n";
        return this;
    }


    public Rows query() throws YosException {
        query_str = "from(bucket: \"" + siteid + "\")\n" +
                this.range_str +
                "  |> filter(fn: (r) => r[\"_measurement\"] == \"" + w_productid + "\")\n" +
                "  |> filter(fn: (r) => r[\"deviceid\"] == \"" + w_deviceid + "\")\n" +
                "  |> filter(fn: (r) => r[\"_field\"] == \"" + param + "\")\n";
        return this.getRows();
    }

    public Rows query(FN fn) throws YosException {
        query_str = "from(bucket: \"" + siteid + "\")\n" +
                this.range_str +
                "  |> filter(fn: (r) => r[\"_measurement\"] == \"" + w_productid + "\")\n" +
                "  |> filter(fn: (r) => r[\"deviceid\"] == \"" + w_deviceid + "\")\n" +
                "  |> filter(fn: (r) => r[\"_field\"] == \"" + param + "\")\n" +
                "  |> " + fn.name() + "()";
        return this.getRows();
    }

    private TimeType fun_timetype;

    public Rows query(FN fn, TimeType timetype, int timespan) throws YosException {
        this.fun_timetype = timetype;
        query_str = "from(bucket: \"" + siteid + "\")\n" +
                this.range_str +
                "  |> filter(fn: (r) => r[\"_measurement\"] == \"" + w_productid + "\")\n" +
                "  |> filter(fn: (r) => r[\"deviceid\"] == \"" + w_deviceid + "\")\n" +
                "  |> filter(fn: (r) => r[\"_field\"] == \"" + param + "\")\n" +
                "  |> window(every: " + timespan + timetype.name() + ")" +
                "  |> " + fn.name() + "()";
        return this.getRows();
    }

    public Rows query(String flux) throws YosException {
        query_str = flux;
        return this.getRows();
    }

    private Rows getRows() throws YosException {
        System.out.println(query_str);
        List<FluxTable> list = InfluxDBConnect.runSqlQuery(siteid, query_str);
        Rows rows = new Rows();
        for (FluxTable fluxTable : list) {
            for (FluxRecord fluxRecord : fluxTable.getRecords()) {
                Row row = new Row();
                row.put("field", fluxRecord.getField());
                row.put("value", fluxRecord.getValue());
                Instant time;
                if ((time = fluxRecord.getTime()) != null) {
                    switch (fun_timetype) {
                        case y: {
                            row.put("time", getDateTime_Str(time.toEpochMilli(), "yyyy"));
                            break;
                        }
                        case mo: {
                            row.put("time", getDateTime_Str(time.toEpochMilli(), "yyyy-MM"));
                            break;
                        }
                        case d: {
                            row.put("time", getDateTime_Str(time.toEpochMilli(), "yyyy-MM-dd"));
                            break;
                        }
                        case h: {
                            row.put("time", getDateTime_Str(time.toEpochMilli(), "yyyy-MM-dd HH"));
                            break;
                        }
                        case m: {
                            row.put("time", getDateTime_Str(time.toEpochMilli(), "yyyy-MM-dd HH:mm"));
                            break;
                        }
                        default: {
                            row.put("time", getDateTime_Str(time.toEpochMilli(), "yyyy-MM-dd HH:mm:ss"));
                            break;
                        }
                    }
                }
                Instant start;
                if ((start = fluxRecord.getStart()) != null) {
                    row.put("start", getDateTime_Str(start.toEpochMilli(), "yyyy-MM-dd HH:mm:ss"));
                }
                Instant stop;
                if ((stop = fluxRecord.getStop()) != null) {
                    row.put("stop", getDateTime_Str(stop.toEpochMilli(), "yyyy-MM-dd HH:mm:ss"));
                }
                rows.add(row);
            }
        }
        return rows;
    }

    public static void main(String[] args) throws YosException {
        Rows rows = new InfluxDBQuery("CS", 22371, 7L, "DownP").query();
        System.err.println(rows.toJsonArray());
    }

    private String getDateTime_Str(Long imeinmillis, String format_str) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeInMillis(imeinmillis);
        SimpleDateFormat format = new SimpleDateFormat(format_str);
        return format.format(calendar.getTime());
    }


    public enum TimeType {
        /**
         * 年
         */
        y,
        /**
         * 月
         */
        mo,
        /**
         * 日
         */
        d,
        /**
         * 小时
         */
        h,
        /**
         * 分钟
         */
        m,
        /**
         * 秒
         */
        s
    }

    public enum FN {
        sum, mean, first, last, max, min
    }

}
