/**
 *
 */
package common.data.db;

import com.alibaba.fastjson.JSON;
import common.YosException;
import common.data.FieldMeta;
import common.data.Row;
import common.data.Rows;
import common.data.SQLFactory;

import java.math.BigDecimal;
import java.sql.*;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;

/**
 * @author 沈静伟
 *
 * 获取一个数据库连接
 */
public class DBConnect {

    private boolean createsqllog = false;
    /**
     * 数据库连接名称
     */
    private String databasekey;
    private ArrayList<SQLLog> sqlLogList = new ArrayList<>();

    /**
     * 默认构造函数，databasekey=default 表示此次获取yos系统数据库连接
     */
    public DBConnect() {
        this.databasekey = "default";
    }

    /**
     * 获取指定的数据库连接，databasekey需要在yos中注册
     * @param databasekey
     */
    public DBConnect(String databasekey) {
        this.databasekey = databasekey;
    }

    /**
     * 获取数据库名称
     * @return
     */
    public String getDBName() throws YosException {
        if (this.databasekey.equals("default")) {
            return runSqlQuery(0, "select database() as dbname").getString("dbname");
        } else {
            return runSqlQuery(0, "select dbname from sys_datasource where datakey='" + databasekey + "'").getString("dbname");
        }
    }

    /**
     * 是否需要记录sql日志
     * @param createsqllog
     */
    public void setCreatesqllog(boolean createsqllog) {
        this.createsqllog = createsqllog;
    }

    /**
     * 获取SQL执行日志
     * @return
     */
    public ArrayList<SQLLog> getSqlLogList() {
        return sqlLogList;
    }

    /**
     * 执行sql查询语句，并获取指定行的数据
     * @param index
     * @param sqlFactory
     * @return
     */
    public Row runSqlQuery(int index, SQLFactory sqlFactory) throws YosException {
        return runSqlQuery(sqlFactory).get(index);
    }

    /**
     * 执行sql查询语句，并获取指定行的数据
     * @param index
     * @param SQL
     * @return
     */
    public Row runSqlQuery(int index, String SQL) throws YosException {
        return runSqlQuery(SQL).get(index);
    }

    /**
     * 执行sql查询语句
     * @param sqlFactory
     * @return
     */
    public Rows runSqlQuery(SQLFactory sqlFactory) throws YosException {
        return runSqlQuery(sqlFactory.getSQL());
    }

    /**
     * 执行sql查询语句
     * @param SQL
     * @return
     */
    public Rows runSqlQuery(String SQL) throws YosException {
        long starttimes = Calendar.getInstance().getTimeInMillis();
        Connection conn = new DBConnectPool().getConnect(databasekey);
        if (conn == null) {
            if (createsqllog) {
                sqlLogList.add(new SQLLog(SQL, databasekey + "连接失败"));
            }
            throw new YosException(databasekey + "连接失败");
        }
        Rows rows = new Rows();
        ResultSet resultSet = null;
        PreparedStatement statement = null;
        try {
            statement = conn.prepareStatement(SQL);
            resultSet = statement.executeQuery();
            ResultSetMetaData rsmd = resultSet.getMetaData();
            int colCount = rsmd.getColumnCount();

            ArrayList<String> colNameList = new ArrayList<String>();
            HashMap<String, FieldMeta> getFieldMetaMap = new HashMap<>();

            for (int i = 0; i < colCount; i++) {

                FieldMeta fieldmeta = new FieldMeta();
                fieldmeta.setTable_name(rsmd.getTableName(i + 1));
                fieldmeta.setColumn_name(rsmd.getColumnName(i + 1));
                fieldmeta.setPrecision(rsmd.getPrecision(i + 1));
                fieldmeta.setScale(rsmd.getScale(i + 1));

                String Columnlabel = rsmd.getColumnLabel(i + 1);
                colNameList.add(Columnlabel);

                switch (rsmd.getColumnTypeName(i + 1).toLowerCase()) {
                    case "int":
                    case "smallint": {
                        fieldmeta.setFieldtype(Integer.class);
                        break;
                    }
                    case "bigint": {
                        fieldmeta.setFieldtype(Long.class);
                        break;
                    }
                    case "float":
                    case "numeric":
                    case "decimal": {
                        fieldmeta.setFieldtype(BigDecimal.class);
                        break;
                    }
                    case "date":
                    case "datetime": {
                        fieldmeta.setFieldtype(Date.class);
                        break;
                    }
                    case "boolean": {
                        fieldmeta.setFieldtype(Boolean.class);
                        break;
                    }
                    case "json": {
                        fieldmeta.setFieldtype(JSON.class);
                        break;
                    }
                    default: {
                        fieldmeta.setFieldtype(String.class);
                        break;
                    }
                }
                getFieldMetaMap.put(Columnlabel, fieldmeta);
            }
            boolean addrowindex = !colNameList.contains("rowindex");
            int rowindex = 1;
            while (resultSet.next()) {
                Row row = new Row();
                for (int i = 0; i < colCount; i++) {
                    String key = colNameList.get(i);
                    Class clazz = getFieldMetaMap.get(key).getFieldtype();
                    Object value;
                    if (clazz == Integer.class) {
                        value = resultSet.getInt(colNameList.get(i));
                    } else if (clazz == Long.class) {
                        value = resultSet.getLong(colNameList.get(i));
                    } else if (clazz == BigDecimal.class) {
                        value = resultSet.getBigDecimal(colNameList.get(i));
                    } else if (clazz == Boolean.class) {
                        value = resultSet.getBoolean(colNameList.get(i));
                    } else if (clazz == JSON.class) {
                        value = JSON.parse(resultSet.getString(colNameList.get(i)));
                    } else if (clazz == Date.class) {
                        value = resultSet.getString(colNameList.get(i));
                        if (value != null) {
                            value = ((String) value).replace(" 00:00:00.0", "");
                        }
                    } else {
                        Object objvalue = resultSet.getObject(colNameList.get(i));
                        value = (objvalue == null) ? "" : objvalue;
                    }
                    if ("total".equals(key)) {
                        rows.totalRows = (Long) value;
                    } else if ("pageTotal".equals(key)) {
                        rows.totalPage = ((BigDecimal) value).longValue();
                    } else {
                        row.put(key, value);
                    }
                }
                if (addrowindex) {
                    row.put("rowindex", rowindex++);
                }
                rows.add(row);
            }
            rows.setFieldList(colNameList);
            rows.setFieldMetaMap(getFieldMetaMap);

            if (rows.totalRows == 0) {
                rows.totalRows = rows.size();
                rows.totalPage = 1;
            }
        } catch (Exception e) {
            try {
                conn.close();
                conn = null;
            } catch (Exception e1) {
                e1.printStackTrace();
            }
            if (createsqllog) {
                sqlLogList.add(new SQLLog(SQL, databasekey + "查询语句执行错误：" + e.getMessage()));
            }
            throw new YosException(databasekey + "查询语句执行错误：" + SQL + System.lineSeparator() + e.getMessage());
        } finally {
            if (resultSet != null) {
                try {
                    resultSet.close();
                } catch (SQLException e) {
                    throw new YosException(e.getMessage());
                }
            }
            close(conn, statement);
        }
        if (createsqllog) {
            sqlLogList.add(new SQLLog(SQL, Calendar.getInstance().getTimeInMillis() - starttimes));
        }
        return rows;
    }

    /**
     * 执行sql更新语句
     * @param sqlFactory
     * @return
     */
    public void runSqlUpdate(SQLFactory sqlFactory) throws YosException {
        runSqlUpdate(sqlFactory.getSQL());
    }

    /**
     * 执行sql更新语句
     * @param SQL
     * @return
     */
    public void runSqlUpdate(String SQL) throws YosException {
        long starttimes = Calendar.getInstance().getTimeInMillis();
        Connection conn = new DBConnectPool().getConnect(databasekey);
        if (conn == null) {
            if (createsqllog) {
                sqlLogList.add(new SQLLog(SQL, databasekey + "连接失败"));
            }
            throw new YosException(databasekey + "连接失败");
        }
        PreparedStatement statement = null;
        try {
            statement = conn.prepareStatement(SQL);
            statement.executeUpdate();
            conn.commit();
        } catch (Exception e) {
            try {
                conn.rollback();
            } catch (SQLException e1) {
                throw new YosException(e1.getMessage());
            }
            try {
                conn.close();
                conn = null;
            } catch (SQLException e1) {
                throw new YosException(e1.getMessage());
            }
            if (createsqllog) {
                sqlLogList.add(new SQLLog(SQL, databasekey + "更新语句执行错误：" + e.getMessage()));
            }
            throw new YosException(databasekey + "更新语句执行错误：" + SQL + System.lineSeparator() + e.getMessage());
        } finally {
            close(conn, statement);
        }
        if (createsqllog) {
            sqlLogList.add(new SQLLog(SQL, Calendar.getInstance().getTimeInMillis() - starttimes));
        }
    }

    /**
     * 执行sql更新语句组
     * @param SQLlist
     * @return
     */
    public void runSqlUpdate(ArrayList<String> SQLlist) throws YosException {
        long starttimes = Calendar.getInstance().getTimeInMillis();
        Connection conn = new DBConnectPool().getConnect(databasekey);
        if (conn == null) {
            if (createsqllog) {
                for (String SQL : SQLlist) {
                    sqlLogList.add(new SQLLog(SQL, databasekey + "连接失败"));
                }
            }
            throw new YosException(databasekey + "连接失败");
        }
        PreparedStatement statement = null;
        try {
            for (String sql : SQLlist) {
                statement = conn.prepareStatement(sql);
                statement.executeUpdate();
            }
            conn.commit();
        } catch (Exception e) {
            try {
                conn.rollback();
            } catch (SQLException e1) {
                throw new YosException(e1.getMessage());
            }
            try {
                conn.close();
                conn = null;
            } catch (SQLException e1) {
                throw new YosException(e1.getMessage());
            }
            if (createsqllog) {
                for (String SQL : SQLlist) {
                    sqlLogList.add(new SQLLog(SQL, databasekey + "更新语句执行错误：" + e.getMessage()));
                }
            }
            throw new YosException(databasekey + "更新语句执行错误：" + JSON.toJSON(SQLlist) + System.lineSeparator() + e.getMessage());
        } finally {
            close(conn, statement);
        }
        if (createsqllog) {
            for (String SQL : SQLlist) {
                sqlLogList.add(new SQLLog(SQL, Calendar.getInstance().getTimeInMillis() - starttimes));
            }
        }
    }

    /**
     * 数据库连接释放
     * @param conn
     * @param statement
     */
    private void close(Connection conn, PreparedStatement statement) {
        if (statement != null) {
            try {
                statement.close();
                statement = null;
            } catch (SQLException e) {
                e.printStackTrace();
            }
        }
        if (conn != null) {
            new DBConnectPool().close(databasekey, conn);
        }
    }

    public String getDateTime_Str() {
        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return format.format(calendar.getTime());
    }

}
